<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\InsuranceBeneficiary;
use App\Models\InsuranceSubscription;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Validator;


class InsuranceBeneficiaryController extends Controller
{
    /**
     * Add a beneficiary
     */
// public function store(Request $request)
//     {
//         $request->validate([
//             'subscription_id' => 'required|exists:insurance_subscriptions,id',
//             'name' => 'required|string',
//             'relation' => 'required|in:father,mother,spouse,child',
//             'cnic' => 'nullable|string',
//             'dob' => 'nullable|date',
//             'gender' => 'nullable|in:male,female,other',
//         ]);

//         $subscription = InsuranceSubscription::findOrFail($request->subscription_id);

//         // Fetch current beneficiaries
//         $beneficiaries = $subscription->beneficiaries;

//         // Rule 1: Max total beneficiaries = 4
//         if ($beneficiaries->count() >= 4) {
//             return response()->json([
//                 'message' => 'Maximum 4 beneficiaries allowed (Spouse, Parents, 2 Children).'
//             ], 422);
//         }

//         // Rule 2: Only one spouse
//         if ($request->relation === 'spouse' && $beneficiaries->where('relation', 'spouse')->count() >= 1) {
//             return response()->json([
//                 'message' => 'Only one spouse can be added.'
//             ], 422);
//         }

//         // Rule 3: Only one father
//         if ($request->relation === 'father' && $beneficiaries->where('relation', 'father')->count() >= 1) {
//             return response()->json([
//                 'message' => 'Only one father can be added.'
//             ], 422);
//         }

//         // Rule 4: Only one mother
//         if ($request->relation === 'mother' && $beneficiaries->where('relation', 'mother')->count() >= 1) {
//             return response()->json([
//                 'message' => 'Only one mother can be added.'
//             ], 422);
//         }

//         // Rule 5: Max 2 children
//         if ($request->relation === 'child' && $beneficiaries->where('relation', 'child')->count() >= 2) {
//             return response()->json([
//                 'message' => 'Maximum 2 children can be added.'
//             ], 422);
//         }

//         // ✅ Passed all rules → Create beneficiary
//         $beneficiary = InsuranceBeneficiary::create($request->all());

//         return response()->json([
//             'message' => 'Beneficiary added successfully',
//             'beneficiary' => $beneficiary
//         ], 201);
//     }



public function store(Request $request)
{
    $request->validate([
        'subscription_id' => 'required|exists:insurance_subscriptions,id',
        'beneficiaries' => 'required|array|min:1|max:4',
        'beneficiaries.*.name' => 'required|string',
        'beneficiaries.*.relation' => 'required|in:father,mother,spouse,child',
        'beneficiaries.*.cnic' => 'nullable|string',
        'beneficiaries.*.dob' => 'nullable|date',
        'beneficiaries.*.gender' => 'nullable|in:male,female,other',
    ]);

    $subscription = InsuranceSubscription::findOrFail($request->subscription_id);
    $existing = $subscription->beneficiaries;

    // Rule 1: Max total beneficiaries = 4
    if ($existing->count() + count($request->beneficiaries) > 4) {
        return response()->json(['message' => 'Maximum 4 beneficiaries allowed.'], 422);
    }

    foreach ($request->beneficiaries as $beneficiaryData) {
        $relation = $beneficiaryData['relation'];

        // Rule 2: Only one spouse
        if ($relation === 'spouse' && $existing->where('relation', 'spouse')->count() >= 1) {
            return response()->json(['message' => 'Only one spouse can be added.'], 422);
        }

        // Rule 3: Only one father
        if ($relation === 'father' && $existing->where('relation', 'father')->count() >= 1) {
            return response()->json(['message' => 'Only one father can be added.'], 422);
        }

        // Rule 4: Only one mother
        if ($relation === 'mother' && $existing->where('relation', 'mother')->count() >= 1) {
            return response()->json(['message' => 'Only one mother can be added.'], 422);
        }

        // Rule 5: Max 2 children
        if ($relation === 'child' && $existing->where('relation', 'child')->count() >= 2) {
            return response()->json(['message' => 'Maximum 2 children can be added.'], 422);
        }

        // Create record and push to collection for next checks
        $created = InsuranceBeneficiary::create([
            'subscription_id' => $request->subscription_id,
            'name' => $beneficiaryData['name'],
            'relation' => $beneficiaryData['relation'],
            'cnic' => $beneficiaryData['cnic'] ?? null,
            'dob' => $beneficiaryData['dob'] ?? null,
            'gender' => $beneficiaryData['gender'] ?? null,
        ]);

        $existing->push($created); // Add to existing list to maintain live validation
    }

    return response()->json(['message' => 'Beneficiaries added successfully'], 201);
}


    /**
     * List beneficiaries for a subscription
     */
    public function list($subscriptionId)
    {
        $beneficiaries = InsuranceBeneficiary::where('subscription_id', $subscriptionId)->get();

        return response()->json($beneficiaries);
    }
    
 
public function beneficiariesdetails(Request $request)
{
    
    // dd('hi');
    try {
        // Step 1: Manual validation (so we can catch inside try)
        $validator = Validator::make($request->all(), [
            'beneficiary_id' => 'required|integer|exists:insurance_beneficiaries,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors(),
            ], 422);
        }

        // Step 2: Get ID
        $beneficiaryId = $request->beneficiary_id;

        // Step 3: Cache key
        $cacheKey = 'beneficiary_details_' . $beneficiaryId;

        // Step 4: Get data with cache (10 min)
        $beneficiary = Cache::remember($cacheKey, now()->addMinutes(10), function () use ($beneficiaryId) {
            return InsuranceBeneficiary::find($beneficiaryId);
        });

        // Step 5: Return success response
        return response()->json([
            'status' => true,
            'message' => Cache::has($cacheKey)
                ? 'Data fetched from cache'
                : 'Data fetched from database',
            'data' => $beneficiary,
        ]);

    } catch (\Exception $e) {
        // Step 6: Catch unexpected errors
        return response()->json([
            'status' => false,
            'message' => 'Something went wrong',
            'error' => $e->getMessage(),
        ], 500);
    }
}
    
    
}
