<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\InsuranceBeneficiary;
use App\Models\InsuranceSubscription;

class InsuranceBeneficiaryController extends Controller
{
    /**
     * Add a beneficiary
     */
public function store(Request $request)
    {
        $request->validate([
            'subscription_id' => 'required|exists:insurance_subscriptions,id',
            'name' => 'required|string',
            'relation' => 'required|in:father,mother,spouse,child',
            'cnic' => 'nullable|string',
            'dob' => 'nullable|date',
            'gender' => 'nullable|in:male,female,other',
        ]);

        $subscription = InsuranceSubscription::findOrFail($request->subscription_id);

        // Fetch current beneficiaries
        $beneficiaries = $subscription->beneficiaries;

        // Rule 1: Max total beneficiaries = 4
        if ($beneficiaries->count() >= 4) {
            return response()->json([
                'message' => 'Maximum 4 beneficiaries allowed (Spouse, Parents, 2 Children).'
            ], 422);
        }

        // Rule 2: Only one spouse
        if ($request->relation === 'spouse' && $beneficiaries->where('relation', 'spouse')->count() >= 1) {
            return response()->json([
                'message' => 'Only one spouse can be added.'
            ], 422);
        }

        // Rule 3: Only one father
        if ($request->relation === 'father' && $beneficiaries->where('relation', 'father')->count() >= 1) {
            return response()->json([
                'message' => 'Only one father can be added.'
            ], 422);
        }

        // Rule 4: Only one mother
        if ($request->relation === 'mother' && $beneficiaries->where('relation', 'mother')->count() >= 1) {
            return response()->json([
                'message' => 'Only one mother can be added.'
            ], 422);
        }

        // Rule 5: Max 2 children
        if ($request->relation === 'child' && $beneficiaries->where('relation', 'child')->count() >= 2) {
            return response()->json([
                'message' => 'Maximum 2 children can be added.'
            ], 422);
        }

        // ✅ Passed all rules → Create beneficiary
        $beneficiary = InsuranceBeneficiary::create($request->all());

        return response()->json([
            'message' => 'Beneficiary added successfully',
            'beneficiary' => $beneficiary
        ], 201);
    }

    /**
     * List beneficiaries for a subscription
     */
    public function list($subscriptionId)
    {
        $beneficiaries = InsuranceBeneficiary::where('subscription_id', $subscriptionId)->get();

        return response()->json($beneficiaries);
    }
}
