<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Prescription;
use App\Models\Doctor;

class PrescriptionController extends Controller
{
    /**
     * 1. List all prescriptions for a customer by MSISDN
     */
    public function list(Request $request)
{
    $request->validate([
        'msisdn' => 'required|string'
    ]);

    $prescriptions = Prescription::with('doctor')
        ->where('msisdn', $request->msisdn)
        ->orderBy('date_time', 'desc')
        ->get()
        ->map(function ($prescription) {
            return [
                'prescription_number' => $prescription->id,
                'type' => $prescription->type, // self_uploaded / doctor_provided
                'date_time' => $prescription->date_time,
                'doctor_name' => $prescription->doctor?->name,
            ];
        });

    return response()->json($prescriptions);
}


    /**
     * 2. Upload self_uploaded prescription
     */
    public function uploadSelfPrescription(Request $request)
    {
        $request->validate([
            'msisdn' => 'required|string',
            'current_complaints' => 'required|string',
            'image' => 'required|image|max:2048',
        ]);

        $imagePath = $request->file('image')->store('prescriptions', 'public');

        $prescription = Prescription::create([
            'msisdn' => $request->msisdn,
            'type' => 'self_uploaded',
            'current_complaints' => $request->current_complaints,
            'image' => $imagePath,
            'date_time' => now(),
        ]);

        return response()->json([
            'message' => 'Self prescription uploaded successfully',
            'prescription' => $prescription
        ], 201);
    }

    /**
     * 3. Get doctor_provided prescription detail
     */
    public function doctorPrescriptionDetail($id)
{
    $prescription = Prescription::with(['doctor', 'medicines'])
        ->where('id', $id)
        ->where('type', 'doctor_provided')
        ->firstOrFail();

    return response()->json([
        'id' => $prescription->id,
        'msisdn' => $prescription->msisdn,
        'prescription_number' => $prescription->prescription_number,
        'doctor_name' => $prescription->doctor?->name,
        'medications' => $prescription->medicines->map(function ($medicine) {
            return [
                'medicine_name' => $medicine->medicine_name,
                'dosage' => $medicine->dosage,
                'days' => $medicine->days,
                'timing' => $medicine->timing,
                'time_of_day' => $medicine->time_of_day,
            ];
        }),
        'doctor_comment' => $prescription->doctor_comment,
        'next_follow_up' => $prescription->next_follow_up,
        'date_time' => $prescription->date_time,
    ]);
}

}
