<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\CustomerClaim;
use Illuminate\Http\JsonResponse;
use App\Models\Product;
use App\Models\Benefit;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;

class CustomerClaimController extends Controller
{
public function dashboard($msisdn): JsonResponse
{
    $claims = CustomerClaim::with(['product.benefits', 'benefits'])
        ->where('customer_msisdn', $msisdn)
        ->get();

    $grouped = $claims->groupBy(fn($claim) => $claim->product->id);

    $data = $grouped->map(function ($claimsGroup) {
        $product = $claimsGroup->first()->product;

        $benefits = collect();

        foreach ($claimsGroup as $claim) {
            foreach ($claim->benefits as $benefit) {
                $total_amount = $product->benefits
                    ->firstWhere('id', $benefit->id)
                    ->pivot
                    ->coverage_limit ?? 0;

                if (!$benefits->has($benefit->id)) {
                    $benefits->put($benefit->id, [
                        'name' => $benefit->name,
                        'total_amount' => $total_amount,
                        'utilized_amount' => $benefit->pivot->approved_amount ?? 0,
                    ]);
                } else {
                    $benefits[$benefit->id]['utilized_amount'] += $benefit->pivot->approved_amount ?? 0;
                }
            }
        }

        return [
            'product_name' => $product->product_name,
            'benefits' => $benefits->values(),
        ];
    });

    return response()->json([
        'status' => true,
        'message' => 'Customer claims dashboard retrieved successfully',
        'data' => $data->values(),
    ]);
}




    public function submit(Request $request): JsonResponse
    {
        // Validate request
        $validator = Validator::make($request->all(), [
            'customer_msisdn' => 'required|string|max:15',
            'subscription_id' => 'required|integer|exists:insurance_subscriptions,id',
            'product_id'      => 'required|integer|exists:products,id',
            'images.*'        => 'required|image|mimes:jpeg,png,jpg,gif|max:5120', // max 5MB per image
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status'  => false,
                'message' => $validator->errors()->first(),
                'data'    => []
            ], 422);
        }

        // Create claim
        $claim = CustomerClaim::create([
            'claim_id'         => 'CLM-' . time(),
            'customer_msisdn'  => $request->customer_msisdn,
            'subscription_id'  => $request->subscription_id,
            'product_id'       => $request->product_id,
            'status'           => 'PENDING',
            'submitted_at'     => now(),
        ]);

        // Handle image uploads
        $images = [];
        if ($request->hasFile('images')) {
            foreach ($request->file('images') as $image) {
                $path = $image->store('claims', 'public');
                $images[] = $path;
            }
            $claim->images = $images;
            $claim->save();
        }

        // Attach benefits (optional: initialize utilized_amount to 0)
        $product = Product::find($request->product_id);
        if ($product && $product->benefits) {
            foreach ($product->benefits as $benefit) {
                $claim->benefits()->attach($benefit->id, ['approved_amount' => 0]);
            }
        }

        return response()->json([
            'status'  => true,
            'message' => 'Claim submitted successfully',
            'data'    => [
                'claim_id' => $claim->claim_id,
                'status'   => $claim->status,
                'images'   => $images
            ]
        ]);
    }



    public function listClaims(Request $request, $msisdn): JsonResponse
{
    // Validate filter
    $status = $request->query('status'); // approved, pending, etc.

    $query = CustomerClaim::with('product')
        ->where('customer_msisdn', $msisdn);

    if ($status) {
        $query->where('status', strtoupper($status)); // make sure status matches DB
    }

    $claims = $query->get();

    $data = $claims->map(function ($claim) {
        return [
            'claim_id' => $claim->claim_id,
            'submitted_at' => $claim->submitted_at,
            'product_name' => $claim->product->product_name ?? null,
            'status' => $claim->status,
        ];
    });

    return response()->json([
        'status' => true,
        'message' => 'Customer claims retrieved successfully',
        'data' => $data,
    ]);
}
}
