<?php

namespace App\Http\Controllers\Api;
use App\Http\Controllers\Controller;
use App\Models\InsuranceSubscription;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Http\JsonResponse;

class SubscriptionController extends Controller
{
    // Subscribe API
    public function subscribe(Request $request)
    {
        $request->validate([
            'plan_id' => 'required|integer',
            'product_id' => 'required|integer',
            'amount' => 'required|numeric',
            'payment_frequency' => 'required|in:DAILY,MONTHLY,ANNUAL,INSTALLMENT',
            'customer_msisdn' => 'required|string',
            'customer_name' => 'required|string',
            'customer_cnic' => 'required|string',
            'jazzcash_tid' => 'required|string|unique:insurance_subscriptions',
        ]);

        $subscription = InsuranceSubscription::create([
            'policy_id' => 'INS-' . date('Ymd') . '-' . strtoupper(Str::random(6)),
            'plan_id' => $request->plan_id,
            'product_id' => $request->product_id,
            'amount' => $request->amount,
            'payment_frequency' => $request->payment_frequency,
            'installment_details' => $request->installment_details,
            'customer_msisdn' => $request->customer_msisdn,
            'customer_name' => $request->customer_name,
            'customer_cnic' => $request->customer_cnic,
            'jazzcash_tid' => $request->jazzcash_tid,
            'jazzcash_reference_id' => $request->jazzcash_reference_id,
            'next_charging_date' => now()->addMonth(),
            'status' => 'ACTIVE',
        ]);

        return response()->json([
            'status' => true,
            'message' => 'Subscription created successfully',
            'data' => $subscription
        ], 201);
    }

    // Unsubscribe API
    public function unsubscribe(Request $request, $id)
    {
        $subscription = InsuranceSubscription::findOrFail($id);
        $subscription->status = 'CANCELLED';
        $subscription->save();

        return response()->json([
            'status' => true,
            'message' => 'Subscription cancelled successfully',
            'data' => $subscription
        ]);
    }


       public function activePoliciesByMsisdn(string $msisdn): JsonResponse
    {
        $subscriptions = InsuranceSubscription::where('customer_msisdn', $msisdn)
            ->where('status', 'ACTIVE')
            ->get([
                'id',
                'policy_id',
                'plan_id',
                'product_id',
                'amount',
                'payment_frequency',
                'next_charging_date',
                'status'
            ]);

        return response()->json([
            'status'  => true,
            'message' => 'Active policies retrieved successfully',
            'data'    => $subscriptions
        ]);
    }

    /**
     * Check if a specific product is subscribed by MSISDN
     */
    public function hasProduct(Request $request): JsonResponse
    {
        $request->validate([
            'msisdn'     => 'required|string',
            'product_id' => 'required|integer'
        ]);

        $subscription = InsuranceSubscription::where('customer_msisdn', $request->msisdn)
            ->where('product_id', $request->product_id)
            ->where('status', 'ACTIVE')
            ->first();

        return response()->json([
            'status'  => true,
            'message' => $subscription
                ? 'Customer has an active subscription for this product'
                : 'Customer does not have an active subscription for this product',
            'data'    => $subscription ? $subscription : []
        ]);
    }
}

