<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\Plan;
use Illuminate\Http\JsonResponse;

class ProductController extends Controller
{
    /**
     * Get all products for a specific plan
     */
   public function index($planId): JsonResponse
{
    $plan = Plan::find($planId);

    if (!$plan) {
        return response()->json([
            'status' => false,
            'message' => 'Plan not found',
            'data' => []
        ], 404);
    }

    $products = Product::with('benefits') // eager load benefits
        ->where('plan_id', $planId)
        ->get(['id', 'product_name', 'category', 'product_code', 'price', 'discounted_price', 'status', 'description', 'customers', 'duration']);

    return response()->json([
        'status' => true,
        'message' => 'Products retrieved successfully',
        'data' => $products->map(function ($product) {
            return [
                'id' => $product->id,
                'product_name' => $product->product_name,
                'category' => $product->category,
                'product_code' => $product->product_code,
                'price' => $product->price,
                'discounted_price' => $product->discounted_price,
                'status' => $product->status,
                'description' => $product->description,
                'customers' => $product->customers,
                'duration' => $product->duration,
                'benefits' => $product->benefits->map(function ($benefit) {
                $coverage = $benefit->pivot->coverage_limit;

    // Special handling for "Additional Benefits"
                    if (strtolower(trim($benefit->name)) === 'additional benefits') {
                        $coverage = preg_split('/[;]+/', $coverage); // split by comma or semicolon
                        $coverage = array_map('trim', $coverage);     // clean spaces
                    }

                return [
                    'id'                => $benefit->id,
                    'benefit_code'      => $benefit->benefit_code,
                    'name'              => $benefit->name,
                    'claim_type'        => $benefit->claim_type,
                    'coverage_limit'    => $coverage,
                    'additional_details'=> $benefit->pivot->additional_details,
                ];
}),
            ];
        }),
    ]);
}


 public function getall(): JsonResponse
{
    $plans = Plan::with([
        'products' => function ($query) {
            $query->select(
                'id',
                'plan_id',
                'product_name',
                'category',
                'product_code',
                'health_system_product_code',
                'description',
                'customers',
                'duration',
                'price',
                'discounted_price',
                'status'
            );
        },
        'products.benefits' => function ($query) {
            $query->select(
                'benefits.id',
                'benefits.name',
                'benefits.benefit_code',
                'benefits.claim_type',
                'benefits.description'
            );
        }
    ])->get(['id', 'name', 'plan_code']);

    // Transform the response
    $plans->transform(function ($plan) {
        $plan->products->transform(function ($product) {
            $product->benefits->transform(function ($benefit) {
                return [
                    'id'              => $benefit->id,
                    'name'            => $benefit->name,
                    'benefit_code'    => $benefit->benefit_code,
                    'claim_type'      => $benefit->claim_type,
                    'description'     => $benefit->description,
                    'coverage_limit'  => $benefit->pivot->coverage_limit,
                ];
            });
            return $product;
        });
        return $plan;
    });

    return response()->json([
        'status'  => true,
        'message' => 'Plans with products & benefits retrieved successfully',
        'data'    => $plans
    ]);
}


}
