<?php

namespace App\Http\Controllers\Api;
use App\Http\Controllers\Controller;
use App\Models\InsuranceSubscription;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use App\Models\Product;
use App\Models\Plan;
use App\Models\Voucher;
use App\Models\InsuranceBeneficiary;
use Illuminate\Http\JsonResponse;


class SubscriptionController extends Controller
{
    // Subscribe API
    public function subscribe(Request $request)
    {
        $request->validate([
            'plan_id' => 'required|integer',
            'product_id' => 'required|integer',
            'amount' => 'required|numeric',
            'actual_amount' => 'required|numeric',
            'payment_frequency' => 'required|in:DAILY,MONTHLY,ANNUAL,INSTALLMENT',
            'customer_msisdn' => 'required|string',
            'customer_name' => 'required|string',
            'customer_cnic' => 'required|string',
            'jazzcash_tid' => 'required|string|unique:insurance_subscriptions',
        ]);

        $subscription = InsuranceSubscription::create([
            'policy_id' => 'INS-' . date('Ymd') . '-' . strtoupper(Str::random(6)),
            'plan_id' => $request->plan_id,
            'product_id' => $request->product_id,
            'amount' => $request->amount,
            'payment_frequency' => $request->payment_frequency,
            'installment_details' => $request->installment_details,
            'customer_msisdn' => $request->customer_msisdn,
            'customer_name' => $request->customer_name,
            'customer_cnic' => $request->customer_cnic,
            'jazzcash_tid' => $request->jazzcash_tid,
             'actual_amount' => $request->actual_amount,
            'jazzcash_reference_id' => $request->jazzcash_reference_id,
            'next_charging_date' => now()->addMonth(),
            'status' => 'ACTIVE',
        ]);

        return response()->json([
            'status' => true,
            'message' => 'Subscription created successfully',
            'data' => $subscription
        ], 201);
    }

    // Unsubscribe API
    public function unsubscribe(Request $request, $id)
    {
        $subscription = InsuranceSubscription::findOrFail($id);
        $subscription->status = 'CANCELLED';
        $subscription->save();

        return response()->json([
            'status' => true,
            'message' => 'Subscription cancelled successfully',
            'data' => $subscription
        ]);
    }


    //   public function activePoliciesByMsisdn(string $msisdn): JsonResponse
    // {
        
    //     // dd('hi');
    //     $subscriptions = InsuranceSubscription::where('customer_msisdn', $msisdn)
    //         ->where('status', 'ACTIVE')
    //         ->get([
    //             'id',
    //             'policy_id',
    //             'plan_id',
    //             'product_id',
    //             'amount',
    //             'payment_frequency',
    //             'next_charging_date',
    //             'status'
    //         ]);

    //     return response()->json([
    //         'status'  => true,
    //         'message' => 'Active policies retrieved successfully',
    //         'data'    => $subscriptions
    //     ]);
    // }

// public function activePoliciesByMsisdn(string $msisdn): JsonResponse
// {
//     $subscriptions = InsuranceSubscription::where('customer_msisdn', $msisdn)
//         ->where('status', 'ACTIVE')
//         ->get([
//             'id',
//             'policy_id',
//             'plan_id',
//             'customer_name',
//             'product_id',
//             'amount',
//             'payment_frequency',
//             'next_charging_date',
//             'status'
//         ]);
        
//         $maxBeneficiariesByProduct = [
//     1 => 1, // product_id 1 -> max 1 beneficiary
//     2 => 4, // product_id 2 -> max 4 beneficiaries
//     3 => 6, // product_id 3 -> max 6 beneficiaries
// ];


//     $subscriptions->transform(function ($subscription) use ($maxBeneficiariesByProduct) {
//     // Product + Plan + Benefits
//     $product = Product::with([
//         'plan:id,name,plan_code',
//         'benefits:id,name,benefit_code,claim_type,description'
//     ])->where('id', $subscription->product_id)
//       ->first([
//           'id',
//           'plan_id',
//           'product_name',
//           'category',
//           'product_code',
//           'health_system_product_code',
//           'description',
//           'customers',
//           'duration',
//           'price',
//           'discounted_price',
//           'status'
//       ]);

//     if ($product) {
//         $product->benefits->transform(function ($benefit) {
//             return [
//                 'id'             => $benefit->id,
//                 'name'           => $benefit->name,
//                 'benefit_code'   => $benefit->benefit_code,
//                 'claim_type'     => $benefit->claim_type,
//                 'description'    => $benefit->description,
//                 'coverage_limit' => $benefit->pivot->coverage_limit ?? null,
//             ];
//         });
//     }

//     // Beneficiaries
//     $beneficiaries = InsuranceBeneficiary::where('subscription_id', $subscription->id)
//         ->get([
//             'id',
//             'subscription_id',
//             'name',
//             'cnic',
//             'relation',
//             'dob',
//             'gender'
//         ]);

//     // Max beneficiaries allowed for this product
//     $maxAllowed = $maxBeneficiariesByProduct[$subscription->product_id] ?? 0;

//     // Available count = existing beneficiaries
//     $availableCount = $beneficiaries->count();

//     // Remaining = max - available (minimum 0)
//     $remainingCount = max(0, $maxAllowed - $availableCount);

//     return [
//         'id'                  => $subscription->id,
//         'policy_id'           => $subscription->policy_id,
//         'customer_name'       => $subscription->customer_name,
//         'plan_id'             => $subscription->plan_id,
//         'product_id'          => $subscription->product_id,
//         'amount'              => $subscription->amount,
//         'payment_frequency'   => $subscription->payment_frequency,
//         'next_charging_date'  => $subscription->next_charging_date,
//         'status'              => $subscription->status,
//         'product'             => $product,
//         'Available_beneficiaries' => [
//             'count' => $availableCount,
//             'list'  => $beneficiaries
//         ],
//         'Remaining_beneficiaries' => [
//             'count' => $remainingCount
//         ]
//     ];
// });

//     return response()->json([
//         'status'  => true,
//         'message' => 'Active policies retrieved successfully',
//         'data'    => $subscriptions
//     ]);
// }


// public function activePoliciesByMsisdn(string $msisdn): JsonResponse
// {
//     $subscriptions = InsuranceSubscription::where('customer_msisdn', $msisdn)
//         ->where('status', 'ACTIVE')
//         ->get([
//             'id',
//             'policy_id',
//             'plan_id',
//             'customer_name',
//             'product_id',
//             'amount',
//             'payment_frequency',
//             'next_charging_date',
//             'subscription_timestamp',
//             'status'
//         ]);

//   $maxBeneficiariesByProduct = [
//     1 => 0, // product_id 1 -> max 1 beneficiary (lekin Individual ko exclude kiya, isliye 0 bachega)
//     2 => 3, // product_id 2 -> Spouse + Child + Child
//     3 => 5, // product_id 3 -> Spouse + Child + Child + Parent + Parent
// ];


//     // Slot-based allowed relations (duplicates mean multiple slots)
//     $allowedRelationsByProduct = [
//     1 => [], // koi slot nahi
//     2 => ['Spouse', 'Child_1', 'Child_2'],
//     3 => ['Spouse', 'Child_1', 'Child_2', 'Parent_1', 'Parent_2'],
// ];

//     $subscriptions->transform(function ($subscription) use ($maxBeneficiariesByProduct, $allowedRelationsByProduct) {
//         // Product + Plan + Benefits
//         $product = Product::with([
//             'plan:id,name,plan_code',
//             'benefits:id,name,benefit_code,claim_type,description'
//         ])->where('id', $subscription->product_id)
//           ->first([
//               'id',
//               'plan_id',
//               'product_name',
//               'category',
//               'product_code',
//               'health_system_product_code',
//               'description',
//               'customers',
//               'duration',
//               'price',
//               'discounted_price',
//               'status'
//           ]);

//         if ($product) {
//             $product->benefits->transform(function ($benefit) {
//                 return [
//                     'id'             => $benefit->id,
//                     'name'           => $benefit->name,
//                     'benefit_code'   => $benefit->benefit_code,
//                     'claim_type'     => $benefit->claim_type,
//                     'description'    => $benefit->description,
//                     'coverage_limit' => $benefit->pivot->coverage_limit ?? null,
//                 ];
//             });
//         }

//         // Current beneficiaries
//         $beneficiaries = InsuranceBeneficiary::where('subscription_id', $subscription->id)
//             ->get([
//                 'id',
//                 'subscription_id',
//                 'name',
//                 'cnic',
//                 'relation',
//                 'dob',
//                 'gender'
//             ]);

//         $maxAllowed = $maxBeneficiariesByProduct[$subscription->product_id] ?? 0;
//         $availableCount = $beneficiaries->count();

//         // Remaining count (numeric)
//         $remainingCount = max(0, $maxAllowed - $availableCount);

//         // Get allowed slots for this product (with duplicates)
//         $allowedSlots = $allowedRelationsByProduct[$subscription->product_id] ?? [];

//         // Normalize used relations from DB to match allowed slots capitalization
//         // e.g. db has 'spouse' or 'Spouse' => we convert to 'Spouse' for comparison
//         $usedRelations = $beneficiaries->pluck('relation')
//             ->map(fn($r) => ucfirst(strtolower($r)))
//             ->toArray();

//         // Remove used relations one-by-one from allowedSlots
//         $remainingSlots = $allowedSlots; // copy
//         foreach ($usedRelations as $used) {
//             $index = array_search($used, $remainingSlots);
//             if ($index !== false) {
//                 // remove single occurrence (preserves duplicates correctly)
//                 array_splice($remainingSlots, $index, 1);
//             }
//         }

//         // Ensure we only return as many relation slots as numeric remainingCount
//         $remainingRelationsToReturn = array_values(array_slice($remainingSlots, 0, $remainingCount));

//         return [
//             'id'                    => $subscription->id,
//             'policy_id'             => $subscription->policy_id,
//             'customer_name'         => $subscription->customer_name,
//             'plan_id'               => $subscription->plan_id,
//             'product_id'            => $subscription->product_id,
//             'amount'                => $subscription->amount,
//             'dicount_amount'       => $subscription->dicount_amount,
           
//             'payment_frequency'     => $subscription->payment_frequency,
//             'next_charging_date'    => $subscription->next_charging_date,
//             'policy_subscription_date'    => $subscription->subscription_timestamp,
//             'expiry_date'    => $subscription->next_charging_date,
//             'status'                => $subscription->status,
//             'product'               => $product,
//             'Available_beneficiaries' => [
//                 'count' => $availableCount,
//                 'list'  => $beneficiaries
//             ],
//             'Remaining_beneficiaries' => [
//                 'count'    => $remainingCount,
//                 'relation' => $remainingRelationsToReturn
//             ]
//         ];
//     });

//     return response()->json([
//         'status'  => true,
//         'message' => 'Active policies retrieved successfully',
//         'data'    => $subscriptions
//     ]);
// }

// public function activePoliciesByMsisdn(string $msisdn): JsonResponse
// {
//     // Fetch active subscriptions for the customer
//     $subscriptions = InsuranceSubscription::where('customer_msisdn', $msisdn)
//         ->where('status', 'ACTIVE')
//         ->get([
//             'id',
//             'policy_id',
//             'plan_id',
//             'customer_name',
//             'product_id',
//             'amount',
//             'dicount_amount',
//             'payment_frequency',
//             'next_charging_date',
//             'subscription_timestamp',
//             'status'
//         ]);

//     // Max beneficiaries allowed per product
//     $maxBeneficiariesByProduct = [
//         1 => 0,
//         2 => 3,
//         3 => 5,
//     ];

//     // Slot-based allowed relations
//     $allowedRelationsByProduct = [
//         1 => [],
//         2 => ['Spouse', 'Child_1', 'Child_2'],
//         3 => ['Spouse', 'Child_1', 'Child_2', 'Parent_1', 'Parent_2'],
//     ];

//     // Mapping for relation Urdu translation (static, you can add more)
//     $relationUrTranslations = [
//         'Spouse'   => 'زوجہ',
//         'Child_1'  => 'بچہ',
//         'Child_2'  => 'بچہ',
//         'Parent_1' => 'والد',
//         'Parent_2' => 'والدہ',
//     ];

//     // Helper function for dynamic Urdu translation
//     $translateToUrdu = function (string $text): string {
//         $url = 'https://translate.googleapis.com/translate_a/single?client=gtx&sl=en&tl=ur&dt=t&q=' . urlencode($text);
//         $ch = curl_init($url);
//         curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
//         curl_setopt($ch, CURLOPT_TIMEOUT, 5);
//         $response = curl_exec($ch);
//         curl_close($ch);

//         if ($response) {
//             $data = json_decode($response, true);
//             return $data[0][0][0] ?? $text;
//         }

//         return $text; // fallback if translation fails
//     };

//     // Transform subscriptions
//     $subscriptions->transform(function ($subscription) use ($maxBeneficiariesByProduct, $allowedRelationsByProduct, $relationUrTranslations, $translateToUrdu) {

//         // Fetch product with plan and benefits
//         $product = Product::with(['plan:id,name,plan_code','benefits:id,name,benefit_code,claim_type,description'])
//             ->where('id', $subscription->product_id)
//             ->first([
//                 'id', 'plan_id', 'product_name', 'category', 'product_code',
//                 'health_system_product_code', 'description', 'customers', 'duration',
//                 'price', 'discounted_price', 'status'
//             ]);

//         if ($product) {
//             $product->benefits->transform(function ($benefit) {
//                 return [
//                     'id'             => $benefit->id,
//                     'name'           => $benefit->name,
//                     'benefit_code'   => $benefit->benefit_code,
//                     'claim_type'     => $benefit->claim_type,
//                     'description'    => $benefit->description,
//                     'coverage_limit' => $benefit->pivot->coverage_limit ?? null,
//                 ];
//             });
//         }

//         // Current beneficiaries
//         $beneficiaries = InsuranceBeneficiary::where('subscription_id', $subscription->id)
//             ->get(['id','subscription_id','name','cnic','relation','dob','gender']);

//         $availableCount = $beneficiaries->count();
//         $maxAllowed = $maxBeneficiariesByProduct[$subscription->product_id] ?? 0;
//         $remainingCount = max(0, $maxAllowed - $availableCount);

//         // Calculate remaining relations
//         $allowedSlots = $allowedRelationsByProduct[$subscription->product_id] ?? [];
//         $usedRelations = $beneficiaries->pluck('relation')->map(fn($r) => ucfirst(strtolower($r)))->toArray();
//         $remainingSlots = $allowedSlots;

//         foreach ($usedRelations as $used) {
//             $index = array_search($used, $remainingSlots);
//             if ($index !== false) {
//                 array_splice($remainingSlots, $index, 1);
//             }
//         }

//         $remainingRelationsToReturn = array_values(array_slice($remainingSlots, 0, $remainingCount));
//         $remainingRelationsUr = array_map(fn($r) => $relationUrTranslations[$r] ?? $r, $remainingRelationsToReturn);

//         // Transform beneficiaries with dynamic Urdu translation
//         $beneficiaries = $beneficiaries->map(function ($b) use ($translateToUrdu, $relationUrTranslations) {
//             $relationKey = ucfirst(strtolower($b->relation));
//             return [
//                 'id'              => $b->id,
//                 'subscription_id' => $b->subscription_id,
//                 'name'            => $b->name,
//                 'name_ur'         => $translateToUrdu($b->name),
//                 'cnic'            => $b->cnic,
//                 'relation'        => $b->relation,
//                 'relation_ur'     => $relationUrTranslations[$relationKey] ?? $b->relation,
//                 'dob'             => $b->dob,
//                 'gender'          => $b->gender
//             ];
//         });

//         return [
//             'id'                    => $subscription->id,
//             'policy_id'             => $subscription->policy_id,
//             'customer_name'         => $subscription->customer_name,
//             'plan_id'               => $subscription->plan_id,
//             'product_id'            => $subscription->product_id,
//             'amount'                => $subscription->amount,
//             'dicount_amount'        => $subscription->dicount_amount,
//             'payment_frequency'     => $subscription->payment_frequency,
//             'next_charging_date'    => $subscription->next_charging_date,
//             'policy_subscription_date' => $subscription->subscription_timestamp,
//             'expiry_date'           => $subscription->next_charging_date,
//             'status'                => $subscription->status,
//             'product'               => $product,
//             'Available_beneficiaries'=> [
//                 'count' => $availableCount,
//                 'list'  => $beneficiaries
//             ],
//             'Remaining_beneficiaries'=> [
//                 'count'    => $remainingCount,
//                 'relation' => $remainingRelationsToReturn,
//                 'relation_ur' => $remainingRelationsUr
//             ]
//         ];
//     });

//     return response()->json([
//         'status'  => true,
//         'message' => 'Active policies retrieved successfully',
//         'data'    => $subscriptions
//     ]);
// }

// public function activePoliciesByMsisdn(string $msisdn): JsonResponse
// {
//     // Get active subscriptions
//     $subscriptions = InsuranceSubscription::where('customer_msisdn', $msisdn)
//         ->where('status', 'ACTIVE')
//         ->get([
//             'id',
//             'policy_id',
//             'plan_id',
//             'customer_name',
//             'product_id',
//             'amount',
//             'dicount_amount',
//             'payment_frequency',
//             'next_charging_date',
//             'subscription_timestamp',
//             'status'
//         ]);

//     // Max beneficiaries allowed per product
//     $maxBeneficiariesByProduct = [
//         1 => 0,
//         2 => 3,
//         3 => 5,
//     ];

//     // Slot-based allowed relations
//     $allowedRelationsByProduct = [
//         1 => [],
//         2 => ['Spouse', 'Child_1', 'Child_2'],
//         3 => ['Spouse', 'Child_1', 'Child_2', 'Parent_1', 'Parent_2'],
//     ];

//     // Urdu relation translations
//     $relationUrTranslations = [
//         'Spouse'   => 'زوجہ',
//         'Child_1'  => 'بچہ',
//         'Child_2'  => 'بچہ',
//         'Parent_1' => 'والد',
//         'Parent_2' => 'والدہ',
//     ];

//     /**
//      * UNIVERSAL TRANSLATOR (Auto detect → Urdu or English)
//      */
//     $translateDynamic = function (string $text, string $target): string {

//         if (!$text || trim($text) === "") {
//             return $text;
//         }

//         try {
//             $url = "https://translate.googleapis.com/translate_a/single?client=gtx&sl=auto&tl={$target}&dt=t&q="
//                 . urlencode($text);

//             $response = @file_get_contents($url);
//             $data = json_decode($response, true);

//             return $data[0][0][0] ?? $text;

//         } catch (\Exception $e) {
//             return $text;
//         }
//     };


//     // Transform subscriptions
//     $subscriptions->transform(function ($subscription) use (
//         $maxBeneficiariesByProduct,
//         $allowedRelationsByProduct,
//         $relationUrTranslations,
//         $translateDynamic
//     ) {

//         // Product with plan & benefits
//         $product = Product::with([
//             'plan:id,name,plan_code',
//             'benefits:id,name,benefit_code,claim_type,description'
//         ])
//         ->where('id', $subscription->product_id)
//         ->first([
//             'id','plan_id','product_name','category','product_code',
//             'health_system_product_code','description','customers','duration',
//             'price','discounted_price','status'
//         ]);

//         if ($product) {
//             $product->benefits->transform(function ($benefit) {
//                 return [
//                     'id'             => $benefit->id,
//                     'name'           => $benefit->name,
//                     'benefit_code'   => $benefit->benefit_code,
//                     'claim_type'     => $benefit->claim_type,
//                     'description'    => $benefit->description,
//                     'coverage_limit' => $benefit->pivot->coverage_limit ?? null,
//                 ];
//             });
//         }

//         // Current beneficiaries
//         $beneficiaries = InsuranceBeneficiary::where('subscription_id', $subscription->id)
//             ->get(['id','subscription_id','name','cnic','relation','dob','gender']);

//         $availableCount = $beneficiaries->count();
//         $maxAllowed = $maxBeneficiariesByProduct[$subscription->product_id] ?? 0;
//         $remainingCount = max(0, $maxAllowed - $availableCount);

//         // Calculate remaining slots
//         $allowedSlots = $allowedRelationsByProduct[$subscription->product_id] ?? [];
//         $usedRelations = $beneficiaries->pluck('relation')
//             ->map(fn($r) => ucfirst(strtolower($r)))
//             ->toArray();

//         $remainingSlots = $allowedSlots;

//         foreach ($usedRelations as $used) {
//             $index = array_search($used, $remainingSlots);
//             if ($index !== false) {
//                 array_splice($remainingSlots, $index, 1);
//             }
//         }

//         $remainingRelationsToReturn = array_values(array_slice($remainingSlots, 0, $remainingCount));
//         $remainingRelationsUr = array_map(fn($r) => $relationUrTranslations[$r] ?? $r, $remainingRelationsToReturn);

//         // Translate beneficiary names
//         $beneficiaries = $beneficiaries->map(function ($b) use ($translateDynamic, $relationUrTranslations) {

//             // Auto translate name to Urdu
//             $nameUr = preg_match('/[a-zA-Z]/', $b->name)
//                 ? $translateDynamic($b->name, "ur")
//                 : $b->name;

//             $relationKey = ucfirst(strtolower($b->relation));

//             return [
//                 'id'              => $b->id,
//                 'subscription_id' => $b->subscription_id,
//                 'name'            => $b->name,
//                 'name_ur'         => $nameUr,
//                 'cnic'            => $b->cnic,
//                 'relation'        => $b->relation,
//                 'relation_ur'     => $relationUrTranslations[$relationKey] ?? $b->relation,
//                 'dob'             => $b->dob,
//                 'gender'          => $b->gender
//             ];
//         });

//         // Customer name translation
//         $customerNameUr = preg_match('/[a-zA-Z]/', $subscription->customer_name)
//             ? $translateDynamic($subscription->customer_name, "ur")
//             : $subscription->customer_name;

//         return [
//             'id'                        => $subscription->id,
//             'policy_id'                 => $subscription->policy_id,
//             'customer_name'             => $subscription->customer_name,
//             'customer_name_ur'          => $customerNameUr,
//             'plan_id'                   => $subscription->plan_id,
//             'product_id'                => $subscription->product_id,
//             'amount'                    => $subscription->amount,
//             'dicount_amount'            => $subscription->dicount_amount,
//             'payment_frequency'         => $subscription->payment_frequency,
//             'next_charging_date'        => $subscription->next_charging_date,
//             'policy_subscription_date'  => $subscription->subscription_timestamp,
//             'expiry_date'               => $subscription->next_charging_date,
//             'status'                    => $subscription->status,
//             'product'                   => $product,

//             'Available_beneficiaries' => [
//                 'count' => $availableCount,
//                 'list'  => $beneficiaries
//             ],

//             'Remaining_beneficiaries' => [
//                 'count'       => $remainingCount,
//                 'relation'    => $remainingRelationsToReturn,
//                 'relation_ur' => $remainingRelationsUr
//             ]
//         ];
//     });

//     return response()->json([
//         'status'  => true,
//         'message' => 'Active policies retrieved successfully',
//         'data'    => $subscriptions
//     ]);
// }

public function activePoliciesByMsisdn(string $msisdn): JsonResponse
{
    // Get active subscriptions
    $subscriptions = InsuranceSubscription::where('customer_msisdn', $msisdn)
        ->where('status', 'ACTIVE')
        ->get([
            'id',
            'policy_id',
            'plan_id',
            'customer_name',
            'product_id',
            'amount',
            'dicount_amount',
            'payment_frequency',
            'next_charging_date',
            'subscription_timestamp',
            'status'
        ]);

    // Max beneficiaries allowed per product
    $maxBeneficiariesByProduct = [
        1 => 0,
        2 => 3,
        3 => 5,
    ];

    // Slot-based allowed relations
    $allowedRelationsByProduct = [
        1 => [],
        2 => ['Spouse', 'Child_1', 'Child_2'],
        3 => ['Spouse', 'Child_1', 'Child_2', 'Parent_1', 'Parent_2'],
    ];

    // Urdu relation translations
    $relationUrTranslations = [
        'Spouse'   => 'زوجہ',
        'Child_1'  => 'بچہ',
        'Child_2'  => 'بچہ',
        'Parent_1' => 'والد',
        'Parent_2' => 'والدہ',
    ];

    /**
     * UNIVERSAL TRANSLATOR (Auto detect → Urdu or English)
     */
    $translateDynamic = function (string $text, string $target): string {

        if (!$text || trim($text) === "") {
            return $text;
        }

        try {
            $url = "https://translate.googleapis.com/translate_a/single?client=gtx&sl=auto&tl={$target}&dt=t&q="
                . urlencode($text);

            $response = @file_get_contents($url);
            $data = json_decode($response, true);

            return $data[0][0][0] ?? $text;

        } catch (\Exception $e) {
            return $text;
        }
    };


    // Transform subscriptions
    $subscriptions->transform(function ($subscription) use (
        $maxBeneficiariesByProduct,
        $allowedRelationsByProduct,
        $relationUrTranslations,
        $translateDynamic
    ) {

        // Product with plan & benefits
        $product = Product::with([
            'plan:id,name,plan_code',
            'benefits:id,name,benefit_code,claim_type,description'
        ])
        ->where('id', $subscription->product_id)
        ->first([
            'id','plan_id','product_name','category','product_code',
            'health_system_product_code','description','customers','duration',
            'price','discounted_price','status'
        ]);

        if ($product) {
            $product->benefits->transform(function ($benefit) {
                return [
                    'id'             => $benefit->id,
                    'name'           => $benefit->name,
                    'benefit_code'   => $benefit->benefit_code,
                    'claim_type'     => $benefit->claim_type,
                    'description'    => $benefit->description,
                    'coverage_limit' => $benefit->pivot->coverage_limit ?? null,
                ];
            });
        }

        // Current beneficiaries
        $beneficiaries = InsuranceBeneficiary::where('subscription_id', $subscription->id)
            ->get(['id','subscription_id','name','cnic','relation','dob','gender']);

        $availableCount = $beneficiaries->count();
        $maxAllowed = $maxBeneficiariesByProduct[$subscription->product_id] ?? 0;
        $remainingCount = max(0, $maxAllowed - $availableCount);

        // Calculate remaining slots
        $allowedSlots = $allowedRelationsByProduct[$subscription->product_id] ?? [];
        $usedRelations = [];

        // CHILD RELATION FIXER
        $childIndex = 1; // assign child_1, child_2

        $beneficiaries = $beneficiaries->map(function ($b) use ($translateDynamic, $relationUrTranslations, &$childIndex, &$usedRelations) {

            // Auto translate name to Urdu
            $nameUr = preg_match('/[a-zA-Z]/', $b->name)
                ? $translateDynamic($b->name, "ur")
                : $b->name;

            $relationOriginal = strtolower($b->relation);
            $relationFinal = '';

            // Convert child → child_1 / child_2
            if ($relationOriginal === 'child') {
                $relationFinal = "Child_" . $childIndex;
                $childIndex++;
            } else {
                $relationFinal = ucfirst($relationOriginal); // spouse, parent etc
            }

            $usedRelations[] = $relationFinal;

            return [
                'id'              => $b->id,
                'subscription_id' => $b->subscription_id,
                'name'            => $b->name,
                'name_ur'         => $nameUr,
                'cnic'            => $b->cnic,
                'relation'        => $relationFinal,
                'relation_ur'     => $relationUrTranslations[$relationFinal] ?? $relationFinal,
                'dob'             => $b->dob,
                'gender'          => $b->gender
            ];
        });

        // Calculate remaining slots after child_1 / child_2 fix
        $remainingSlots = array_values(array_diff($allowedSlots, $usedRelations));

        $remainingRelationsToReturn = array_slice($remainingSlots, 0, $remainingCount);
        $remainingRelationsUr = array_map(fn($r) => $relationUrTranslations[$r] ?? $r, $remainingRelationsToReturn);

        // Customer name translation
        $customerNameUr = preg_match('/[a-zA-Z]/', $subscription->customer_name)
            ? $translateDynamic($subscription->customer_name, "ur")
            : $subscription->customer_name;

        return [
            'id'                        => $subscription->id,
            'policy_id'                 => $subscription->policy_id,
            'customer_name'             => $subscription->customer_name,
            'customer_name_ur'          => $customerNameUr,
            'plan_id'                   => $subscription->plan_id,
            'product_id'                => $subscription->product_id,
            'amount'                    => $subscription->amount,
            'dicount_amount'            => $subscription->dicount_amount,
            'payment_frequency'         => $subscription->payment_frequency,
            'next_charging_date'        => $subscription->next_charging_date,
            'policy_subscription_date'  => $subscription->subscription_timestamp,
            'expiry_date'               => $subscription->next_charging_date,
            'status'                    => $subscription->status,
            'product'                   => $product,

            'Available_beneficiaries' => [
                'count' => $availableCount,
                'list'  => $beneficiaries
            ],

            'Remaining_beneficiaries' => [
                'count'       => $remainingCount,
                'relation'    => $remainingRelationsToReturn,
                'relation_ur' => $remainingRelationsUr
            ]
        ];
    });

    return response()->json([
        'status'  => true,
        'message' => 'Active policies retrieved successfully',
        'data'    => $subscriptions
    ]);
}



    /**
     * Check if a specific product is subscribed by MSISDN
     */
    public function hasProduct(Request $request): JsonResponse
    {
        $request->validate([
            'msisdn'     => 'required|string',
            'product_id' => 'required|integer'
        ]);

        $subscription = InsuranceSubscription::where('customer_msisdn', $request->msisdn)
            ->where('product_id', $request->product_id)
            ->where('status', 'ACTIVE')
            ->first();

        return response()->json([
            'status'  => true,
            'message' => $subscription
                ? 'Customer has an active subscription for this product'
                : 'Customer does not have an active subscription for this product',
            'data'    => $subscription ? $subscription : []
        ]);
    }
   
//   public function applyVoucher(Request $request): JsonResponse
// {
//     $subscriptionId = $request->subscription_id;
//     $voucherCode = $request->voucher_code;

//     // Check voucher exists
//     $voucher = Voucher::where('voucher_code', $voucherCode)->first();
//     if (!$voucher) {
//         return response()->json([
//             'status' => false,
//             'message' => 'Voucher code not found'
//         ], 404);
//     }

//     // Check if voucher already used
//     if ($voucher->status === 'used') {
//         return response()->json([
//             'status' => false,
//             'message' => 'Voucher code already used'
//         ], 400);
//     }

//     // Check subscription is active
//     $subscription = InsuranceSubscription::where('id', $subscriptionId)
//         ->where('status', 'ACTIVE')
//         ->first();

//     if (!$subscription) {
//         return response()->json([
//             'status' => false,
//             'message' => 'Subscription not active'
//         ], 400);
//     }

//     // Update voucher with subscription_id and mark as used
//     $voucher->subscription_id = $subscription->id;
//     $voucher->status = 'used';
//     $voucher->save();

//     return response()->json([
//         'status' => true,
//         'message' => 'Voucher applied successfully',
//         'data' => [
//             'subscription_id' => $subscription->id,
//             'voucher_code' => $voucher->voucher_code,
//             'discount_price' => $voucher->discount_price
//         ]
//     ]);
// }
 
public function applyVoucher(Request $request): JsonResponse
{
    try {
        // Validate input
        $request->validate([
            'product_id'   => 'required|integer|exists:products,id',
            'voucher_code' => 'required|string',
            'mode'         => 'required|string|in:Daily,Monthly,Annual' // allowed values
        ]);

        $productId   = $request->product_id;
        $voucherCode = $request->voucher_code;
        $mode        = ucfirst(strtolower($request->mode)); // normalize e.g. daily -> Daily

        // ðŸ”¹ Check voucher exists
        $voucher = Voucher::where('voucher_code', $voucherCode)->first();
        if (!$voucher) {
            return response()->json([
                'status'  => false,
                'message' => 'Voucher code not found'
            ], 404);
        }

        // ðŸ”¹ Check if voucher already used
        if ($voucher->status === 'used') {
            return response()->json([
                'status'  => false,
                'message' => 'Voucher code already used'
            ], 400);
        }

        // ðŸ”¹ Fetch product with modes
        $product = Product::with([
            'plan:id,name,plan_code',
            'benefits:id,name,benefit_code,claim_type,description'
        ])->where('id', $productId)
          ->first([
              'id',
              'plan_id',
              'product_name',
              'category',
              'product_code',
              'health_system_product_code',
              'description',
              'customers',
              'duration',
              'price',
              'discounted_price',
              'status',
              'mode' // store JSON or array
          ]);

        if (!$product) {
            return response()->json([
                'status'  => false,
                'message' => 'Product not found'
            ], 404);
        }

        // ðŸ”¹ Handle mode field (string JSON or already array)
        $modes = is_array($product->mode)
            ? $product->mode
            : json_decode($product->mode, true);

        if (!is_array($modes)) {
            return response()->json([
                'status'  => false,
                'message' => 'Invalid product modes configuration'
            ], 500);
        }

        // ðŸ”¹ Find price for requested mode
        $originalPrice = null;
        foreach ($modes as $item) {
            if (strcasecmp($item['Duration'], $mode) === 0) {
                $originalPrice = (float) $item['Price'];
                break;
            }
        }

        if (is_null($originalPrice)) {
            return response()->json([
                'status'  => false,
                'message' => "Mode '{$mode}' not available for this product"
            ], 400);
        }

       
        $discountPercent = 10;
$discountAmount  = ($originalPrice * $discountPercent) / 100;
$finalPrice      = $originalPrice - $discountAmount;

// ðŸ”¹ Round to nearest integer UP (so 62.1 -> 63)
$finalPrice = ceil($finalPrice);

        // ðŸ”¹ Update voucher
        $voucher->product_id = $product->id;
        $voucher->status     = 'used';
        $voucher->save();

        return response()->json([
            'status'  => true,
            'message' => 'Voucher applied successfully',
            'data'    => [
                'product_id'       => $product->id,
                'voucher_code'     => $voucher->voucher_code,
                'mode'             => $mode,
                'original_price'   => $originalPrice,
                'discount_percent' => $discountPercent . '%',
                'discounted_price' => $finalPrice,
            ]
        ]);

    } catch (\Illuminate\Validation\ValidationException $e) {
        return response()->json([
            'status'  => false,
            'message' => 'Validation failed',
            'errors'  => $e->errors()
        ], 422);

    } catch (\Exception $e) {
        return response()->json([
            'status'  => false,
            'message' => 'Something went wrong',
            'error'   => $e->getMessage()
        ], 500);
    }
}

    
}

