<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\Plan;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;


class ProductController extends Controller
{
    /**
     * Get all products for a specific plan
     */
    public function index($planId, Request $request)
    {
        $lang = $request->header('Accept-Language', 'en');

        $plan = Plan::find($planId);

        if (!$plan) {
            return response()->json([
                'status' => false,
                'message' => 'Plan not found',
                'data' => []
            ], 404);
        }

        // 🔹 Translator function using Google Translate free endpoint
        $translate = function ($text, $targetLang) {
            if ($targetLang === 'ur' && !empty($text)) {
                $url = 'https://translate.googleapis.com/translate_a/single?client=gtx&sl=en&tl=' . $targetLang . '&dt=t&q=' . urlencode($text);
                $response = @file_get_contents($url);
                if ($response) {
                    $result = json_decode($response, true);
                    return $result[0][0][0] ?? $text;
                }
            }
            return $text;
        };

        $products = Product::with('benefits')
            ->where('plan_id', $planId)
            ->get([
                'id',
                'product_name',
                'category',
                'product_code',
                'status',
                'description',
                'customers',
                'duration',
                'mode'
            ]);

        return response()->json([
            'status' => true,
            'message' => 'Products retrieved successfully',
            'data' => $products->map(function ($product) use ($translate, $lang) {
                // 🔸 Translate each "mode" entry
                $translatedModes = collect($product->mode ?? [])->map(function ($m) use ($translate, $lang) {
                    return [
                        'Duration' => $translate($m['Duration'] ?? '', $lang),
                        'Price' => $m['Price'] ?? null,
                        'Discounted_Price' => $m['Discounted_Price'] ?? null,
                    ];
                });

                return [
                    'id' => $product->id,
                    'product_name' => $translate($product->product_name, $lang),
                    'category' => $translate($product->category, $lang),
                    'product_code' => $translate($product->product_code, $lang),
                    'status' => $product->status,
                    'description' => $translate($product->description, $lang),
                    'customers' => $translate($product->customers, $lang),
                    'duration' => $translate($product->duration, $lang),
                    'mode' => $translatedModes,
                    'benefits' => $product->benefits->map(function ($benefit) use ($translate, $lang) {
                        $coverage = $benefit->pivot->coverage_limit;

                        // Special handling for "Additional Benefits"
                        if (strtolower(trim($benefit->name)) === 'additional benefits') {
                            $coverage = preg_split('/[;]+/', $coverage);
                            $coverage = array_map('trim', $coverage);
                        }

                        return [
                            'id' => $benefit->id,
                            'benefit_code' => $benefit->benefit_code,
                            'name' => $translate($benefit->name, $lang),
                            'claim_type' => $translate($benefit->claim_type, $lang),
                            'icon' => $benefit->icon,
                            'coverage_limit' => $coverage,
                            'additional_details' => $benefit->pivot->additional_details,
                        ];
                    }),
                ];
            }),
        ]);
    }


 public function getall(): JsonResponse
{
     //dd('hi');
    $plans = Plan::with([
        'products' => function ($query) {
            $query->select(
                'id',
                'plan_id',
                'product_name',
                'category',
                'product_code',
                'health_system_product_code',
                'description',
                'customers',
                // 'duration',
                // 'price',
                // 'discounted_price',
                'status',
                 'mode' // ✅ add mode here
            );
        },
        
          
        'products.benefits' => function ($query) {
            $query->select(
                'benefits.id',
                'benefits.name',
                'benefits.benefit_code',
                'benefits.claim_type',
                  'benefits.icon',
                'benefits.description'
            );
        }
    ])->get(['id', 'name', 'plan_code']);

    // Transform the response
    $plans->transform(function ($plan) {
        $plan->products->transform(function ($product) {
            $product->benefits->transform(function ($benefit) {
                return [
                    'id'              => $benefit->id,
                    'name'            => $benefit->name,
                    'benefit_code'    => $benefit->benefit_code,
                    'claim_type'      => $benefit->claim_type,
                      'icon'     => $benefit->icon,
                    'description'     => $benefit->description,
                    'coverage_limit'  => $benefit->pivot->coverage_limit,
                ];
            });
            return $product;
        });
        return $plan;
    });

    return response()->json([
        'status'  => true,
        'message' => 'Plans with products & benefits retrieved successfully',
        'data'    => $plans
    ]);
}


}
