<?php

// app/Http/Controllers/Api/OfferController.php
namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Offer;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse; 

class OfferController extends Controller
{
   
   
  public function index(Request $request): JsonResponse
{
    $latitude = $request->get('latitude');
    $longitude = $request->get('longitude');

    $offers = Offer::select(
        'id',
        'name',
        'special_description',
        'rating',
        'discount',
        'latitude',
        'longitude',
        'status',
        'icon'
    )->get();

    // Distance calculation
    foreach ($offers as $offer) {
        $offer->distance = ($latitude && $longitude)
            ? $this->calculateDistance($latitude, $longitude, $offer->latitude, $offer->longitude)
            : null;
    }

    // Google Translate function
    $translate = function ($text) {
        if (!empty($text)) {
            $url = 'https://translate.googleapis.com/translate_a/single?client=gtx&sl=en&tl=ur&dt=t&q=' . urlencode($text);
            $response = @file_get_contents($url);
            if ($response) {
                $result = json_decode($response, true);
                return $result[0][0][0] ?? $text;
            }
        }
        return $text;
    };

    // Build response
    $response = $offers->map(function ($offer) use ($translate) {
        return [
            'id' => $offer->id,
            'name' => $offer->name,
            'special_offer' => $offer->special_description,
            'name_ur' => $translate($offer->name),
            'special_offer_ur' => $translate($offer->special_description),
            'rating' => (string) $offer->rating,
            'distance' => $offer->distance,
            'discount' => $offer->discount . '%',
            'img' => $offer->icon,
        ];
    });

    return response()->json([
        'status' => true,
        'message' => 'Offers retrieved successfully',
        'data' => $response
    ]);
}


        private function calculateDistance($lat1, $lon1, $lat2, $lon2): float
    {
        $earthRadius = 6371; // Radius of Earth in KM

        $latFrom = deg2rad($lat1);
        $lonFrom = deg2rad($lon1);
        $latTo = deg2rad($lat2);
        $lonTo = deg2rad($lon2);

        $latDelta = $latTo - $latFrom;
        $lonDelta = $lonTo - $lonFrom;

        $angle = 2 * asin(sqrt(pow(sin($latDelta / 2), 2) +
                cos($latFrom) * cos($latTo) * pow(sin($lonDelta / 2), 2)));
        return round($earthRadius * $angle, 2); // KM with 2 decimal places
    }

public function show($id): JsonResponse
{
    $offer = Offer::find($id);

    if (!$offer) {
        return response()->json([
            'status' => false,
            'message' => 'Offer not found',
            'data' => []
        ], 404);
    }

    // --- Simple Urdu translation function ---
    $translate = function ($text) {
        if (!empty($text)) {
            $url = 'https://translate.googleapis.com/translate_a/single?client=gtx&sl=en&tl=ur&dt=t&q=' . urlencode($text);
            $response = @file_get_contents($url);
            if ($response) {
                $result = json_decode($response, true);
                return $result[0][0][0] ?? $text;
            }
        }
        return $text;
    };

    // Convert offer to array
    $offerArray = $offer->toArray();

    // Decode offer_detail if JSON
    if (!empty($offerArray['offer_detail'])) {
        $decoded = json_decode($offerArray['offer_detail'], true);

        if (isset($decoded['services'])) {
            $offerArray['offer_detail'] = $decoded['services']; // Only services
        } else {
            $offerArray['offer_detail'] = $decoded;
        }
    }

    // Rename icon to background_image and bk_img
    if (isset($offerArray['icon'])) {
        $offerArray['background_image'] = $offerArray['icon'];
        $offerArray['bk_img'] = $offerArray['icon'];
        unset($offerArray['icon']);
    }

    // --- Urdu Translations ---
    $offerArray['name_ur'] = $translate($offerArray['name'] ?? '');
    $offerArray['special_description_ur'] = $translate($offerArray['special_description'] ?? '');
    $offerArray['type_ur'] = $translate($offerArray['type'] ?? '');
    $offerArray['address_ur'] = $translate($offerArray['address'] ?? '');

    if (!empty($offerArray['offer_detail']) && is_array($offerArray['offer_detail'])) {
        $offerArray['offer_detail_ur'] = array_map(function ($item) use ($translate) {
            return $translate($item);
        }, $offerArray['offer_detail']);
    } else {
        $offerArray['offer_detail_ur'] = [];
    }

    return response()->json([
        'status' => true,
        'message' => 'Offer details retrieved successfully',
        'data' => $offerArray
    ]);
}





}
