<?php

namespace App\Http\Controllers\operations;

use App\Http\Controllers\Controller;
use App\Models\RefundCases;
use App\Models\PdfReport;
use Yajra\DataTables\DataTables;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use App\Models\EasypaisaUser;
use App\Models\InvestmentLedgerSaving;
use Carbon\Carbon;
use App\Models\CustomerSavingsMaster;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use App\Models\InsuranceData;

class OpReportsController extends Controller
{


   public function RefundReport(Request $request)
{
    if ($request->ajax()) {
        $query = RefundCases::with(['customers'])->orderBy('created_at', 'desc');
        
        // dd($query);

        if ($request->from_date && $request->to_date) {
            $query->whereBetween('created_at', [
                $request->from_date . ' 00:00:00',
                $request->to_date . ' 23:59:59'
            ]);
        }

        if ($request->status) {
            $query->where('status', $request->status);
        }

        return Datatables::of($query)
            ->addIndexColumn()
            ->addColumn('user_msisdn', fn($row) => $row->customers->user_msisdn ?? '-')
            ->addColumn('first_name', fn($row) => $row->customers->first_name ?? '-')
            ->addColumn('last_name', fn($row) => $row->customers->last_name ?? '-')
            ->addColumn('email_address', fn($row) => $row->customers->email_address ?? '-')
            ->addColumn('refund_amount', fn($row) => $row->refund_amount)
             ->addColumn('zakat_amount', fn($row) => $row->zakat_amount)
              ->addColumn('profit_amount', fn($row) => $row->profit_amount)
               ->addColumn('refund_request_date', fn($row) => $row->refund_request_date)
                ->addColumn('refund_processed_date', fn($row) => $row->refund_processed_date)
                  ->addColumn('processing_fee_deducted', fn($row) => $row->processing_fee_deducted)
            
           ->addColumn('status', function ($row) {
    $badge = match($row->status) {
        'Accepted' => 'bg-success',
        'Rejected' => 'bg-danger',
        default => 'bg-warning'
    };

    // Show time only for Accepted or Rejected
    $time = in_array($row->status, ['Accepted', 'Rejected']) && $row->status_updated_at
        ? '<br><small>' . \Carbon\Carbon::parse($row->status_updated_at)->format('Y-m-d h:i A') . '</small>'
        : '';

    return '<span class="badge ' . $badge . '">' . $row->status . '</span>' . $time;
})
            ->addColumn('created_at', fn($row) => \Carbon\Carbon::parse($row->created_at)->format('Y-m-d H:i:s'))
            ->addColumn('action', function ($row) {
                if ($row->status === 'Under-Process') {
                    return '
                        <div class="d-flex gap-1">
                            <button class="btn btn-sm btn-success accept-btn" data-id="' . $row->id . '">
                                <i class="fas fa-check"></i> Accept
                            </button>
                            <button class="btn btn-sm btn-danger reject-btn" data-id="' . $row->id . '">
                                <i class="fas fa-times"></i> Reject
                            </button>
                            <a href="/easypaisa/public/storage/uploads/' . $row->file . '" target="_blank" class="btn btn-sm btn-info">
                                <i class="fas fa-file-pdf"></i> View PDF
                            </a>
                        </div>';
                } else {
                    return '
                        <a href="/easypaisa/public/storage/uploads/' . $row->file . '" target="_blank" class="btn btn-sm btn-info">
                            <i class="fas fa-file-pdf"></i> View PDF
                        </a>';
                }
            })
            ->rawColumns(['status', 'action'])
            ->make(true);
    }

    return view('operations.RefundCustomer.index');
}


//     public function updateRefundStatus(Request $request)
// {
//     $refund = RefundCases::findOrFail($request->refund_id);
    
    
    
//     $refund->status = $request->status;
//      $refund->status_updated_at = now();
//     $refund->save();
    
        
//     return response()->json(['message' => 'Status updated']);
// }

public function viewRefundPdf ($id)
{
    $refund = RefundCases::findOrFail($id);

    if (!$refund->file || !Storage::disk('public')->exists($refund->file)) {
        abort(404, 'File not found.');
    }

    $filePath = storage_path('app/public/' . $refund->file);
    return response()->file($filePath);
}

    public function exportRefundReport(Request $request)
    {
        $query = RefundCases::with(['customers'])->orderBy('created_at', 'desc');

        if ($request->from_date && $request->to_date) {
            $query->whereBetween('created_at', [$request->from_date . ' 00:00:00', $request->to_date . ' 23:59:59']);
        }

        if ($request->search_name) {
            $query->whereHas('customers', function ($q) use ($request) {
                $q->where('first_name', 'like', '%' . $request->search_name . '%')
                    ->orWhere('last_name', 'like', '%' . $request->search_name . '%');
            });
        }

        $records = $query->get();

        $filename = 'RefundCustomers_' . now()->format('Y_m_d_H_i_s') . '.csv';

        $headers = [
            "Content-type" => "text/csv",
            "Content-Disposition" => "attachment; filename=$filename",
            "Pragma" => "no-cache",
            "Cache-Control" => "must-revalidate, post-check=0, pre-check=0",
            "Expires" => "0",
        ];

        $callback = function () use ($records) {
            $file = fopen('php://output', 'w');

            // CSV header
            fputcsv($file, [
                'MSISDN',
                'First Name',
                'Last Name',
                'Email',

                'Refund Amount',
                 'Zakat Amount',
                  'Profit Amount',
                   'Refund request date',
                'Status',

                'Date Time',

            ]);

            foreach ($records as $row) {
                fputcsv($file, [
                    $row->customers->user_msisdn ?? '',
                    $row->customers->first_name ?? '',
                    $row->customers->last_name ?? '',
                    $row->customers->email_address ?? '',
                    

                    $row->refund_amount ?? '',
                    
                     $row->zakat_amount ?? '',
                      $row->profit_amount ?? '',
                       $row->refund_request_date ?? '',
                       
                    $row->status ?? '',

                    optional($row->created_at)->format('Y-m-d H:i:s') ?? '',

                ]);
            }


            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }


    public function illustrationReport(Request $request)
    {

        if ($request->ajax()) {
            $query = PdfReport::orderBy('created_at', 'desc');

            if ($request->from_date && $request->to_date) {
                $query->whereBetween('created_at', [$request->from_date . ' 00:00:00', $request->to_date . ' 23:59:59']);
            }



            return Datatables::of($query)
                ->addIndexColumn()

                ->addColumn('name', function ($row) {
                    return $row->name;
                })
                ->addColumn('msisdn', function ($row) {
                    return $row->msisdn;
                })
                ->addColumn('amount', function ($row) {
                    return $row->amount;
                })
                ->addColumn('contribution_term', function ($row) {
                    return $row->contribution_term;
                })
                ->addColumn('sum_covered', function ($row) {
                    return $row->sum_covered;
                })
                ->addColumn('profit_at_9', function ($row) {
                    return $row->profit_at_9;
                })
                ->addColumn('profit_at_13', function ($row) {
                    return $row->profit_at_13;
                })

                ->addColumn('created_at', function ($row) {
                    return \Carbon\Carbon::parse($row->created_at)->format('Y-m-d H:i:s');
                })


                ->rawColumns(['action'])
                ->make(true);
        }


        return view('operations.illustration.index');
    }

    public function exportillustrationReport(Request $request)
    {
        $query = PdfReport::orderBy('created_at', 'desc');

        if ($request->from_date && $request->to_date) {
            $query->whereBetween('created_at', [$request->from_date . ' 00:00:00', $request->to_date . ' 23:59:59']);
        }

        if ($request->search_name) {
            $query->whereHas('customer', function ($q) use ($request) {
                $q->where('first_name', 'like', '%' . $request->search_name . '%')
                    ->orWhere('last_name', 'like', '%' . $request->search_name . '%');
            });
        }

        $records = $query->get();

        $filename = 'illustrationReport_' . now()->format('Y_m_d_H_i_s') . '.csv';

        $headers = [
            "Content-type" => "text/csv",
            "Content-Disposition" => "attachment; filename=$filename",
            "Pragma" => "no-cache",
            "Cache-Control" => "must-revalidate, post-check=0, pre-check=0",
            "Expires" => "0",
        ];

        $callback = function () use ($records) {
            $file = fopen('php://output', 'w');

            fputcsv($file, [
                'Name',
                'MSISDN',
                'Amount',
                'Contribution Term',
                'Sum Covered',
                'Profit at 9%',
                'Profit at 13%',
                'Created At',
            ]);

            // CSV rows
            foreach ($records as $row) {
                fputcsv($file, [
                    $row->name ?? '',
                    $row->msisdn ?? '',
                    $row->amount ?? '',
                    $row->contribution_term ?? '',
                    $row->sum_covered ?? '',
                    $row->profit_at_9 ?? '',
                    $row->profit_at_13 ?? '',
                    \Carbon\Carbon::parse($row->created_at)->format('Y-m-d H:i:s'),
                ]);
            }


            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }


/////

public function updateRefundStatus(Request $request)
{
    $request->validate([
        'refund_id' => 'required|integer|exists:refund_cases,id',
        'status' => 'required|string|in:Accepted,rejected'
    ]);

    DB::beginTransaction();
    try {
        $refund = RefundCases::findOrFail($request->refund_id);

        // Update refund status and timestamp
        $refund->status = $request->status;
        $refund->status_updated_at = now();

        if ($request->status === 'Accepted') {
            // Fetch customer, saving, balance
            $customer = EasypaisaUser::findOrFail($refund->customer_id);
            $saving = CustomerSavingsMaster::findOrFail($refund->saving_id);

            $balance = InvestmentLedgerSaving::where('saving_id', $saving->id)
                ->orderBy('date_time', 'desc')
                ->first();

            // Calculate withdrawal details
            $principalAmount = min($refund->refund_amount, $balance->principal_amount);
            $profitAmount = min($refund->profit_amount, $balance->profit_amount);
            $zakatAmount = 0; // Zakat is exempted after approval
            $zakatOption = '2'; // Declaration form submitted = zakat exempted

            $transactionId = uniqid('txn_');
            $now = now();

            // Fetch EFU Insurance data
            $insuranceData = InsuranceData::where('saving_id', $saving->id)->first();
            if (!$insuranceData || !$insuranceData->customer_id) {
                DB::rollBack();
                return response()->json(['status' => false, 'message' => 'EFU customer ID not found.']);
            }

            $surrenderType = ($balance->principal_amount <= $principalAmount &&
                              $balance->profit_amount <= $profitAmount) ? 'full' : 'partial';

            // EFU API Payload
            $efuPayload = [
                'customer_id' => $insuranceData->eful_policy_number,
                'surrender_type' => $surrenderType,
                'principle_amount' => round($principalAmount, 2),
                'profit_amount' => round($profitAmount, 2),
                'zakat_amount' => round($zakatAmount, 2),
                'zakat_option' => $zakatOption,
            ];

            $efuResponse = Http::withHeaders([
                'Authorization' => 'Bearer XXXXAAA123EPDIGITAKAFUL',
                'channelcode' => 'EPDIGITAKAFUL',
                'Content-Type' => 'application/json',
            ])->post('https://api.efulife.com/epay/digi/tkf/surrender', $efuPayload);

            if (!$efuResponse->successful()) {
                Log::error('EFU API Error', [
                    'payload' => $efuPayload,
                    'response_status' => $efuResponse->status(),
                    'response_body' => $efuResponse->body(),
                ]);
                DB::rollBack();
                return response()->json(['status' => false, 'message' => 'EFU API request failed.']);
            }

            Log::info('EFU API Success', [
                'payload' => $efuPayload,
                'response_status' => $efuResponse->status(),
                'response_body' => $efuResponse->body(),
            ]);

            // Create withdrawal record
            $this->createWithdrawalRecord(
                $customer,
                $saving,
                $transactionId,
                $principalAmount,
                $profitAmount,
                $zakatAmount,
                $balance,
                $now
            );

            // Update saving status
            $this->updateSavingStatus(
                $saving,
                $balance->principal_amount - $principalAmount,
                $principalAmount,
                $profitAmount,
                $zakatAmount
            );

            // Update refund processed date
            $refund->refund_processed_date = now();
        }

        // Save final refund update
        $refund->save();
        DB::commit();

        return response()->json(['status' => true, 'message' => 'Refund status updated and processed successfully.']);
    } catch (\Throwable $e) {
        DB::rollBack();
        return $this->handleError($e);
    }
}


protected function createWithdrawalRecord($customer, $saving, $transactionId, $principalAmount, $profitAmount, $zakatAmount, $balance, $now)
{
    $grossAmount = $principalAmount + $profitAmount;
    $netAmount = $grossAmount - $zakatAmount;

    return InvestmentLedgerSaving::create([
        'customer_id' => $customer->id,
        'saving_id' => $saving->id,
        'customer_msisdn' => $customer->user_msisdn,
        'transaction_id' => $transactionId,
        'transaction_type' => 'withdrawal',
        'amount' => $netAmount,
        'principal_amount' => $balance->principal_amount - $principalAmount,
        'profit_amount' => $balance->profit_amount - $profitAmount,
        'zakat_amount' => $zakatAmount,
        'date_time' => $now,
        'net_amount' => $balance->net_amount - $netAmount,
        'gross_amount' => $grossAmount,
        'breakdown' => [
            'deposit_amount' => $principalAmount,
            'profit_amount' => $profitAmount,
            'zakat_amount' => $zakatAmount,
            'zakat_rate' => '2.5%',
            'zakat_base' => $principalAmount + $profitAmount,
            'transaction_type' => 'withdrawal'
        ]
    ]);
}

protected function updateSavingStatus($saving, $remainingDeposit, $withdrawnPrincipal, $withdrawnProfit, $zakatAmount)
{
    // Update fund growth (deduct only the profit portion)
    $saving->fund_growth_amount = max(0, $saving->fund_growth_amount - $withdrawnProfit - $withdrawnPrincipal);
    
    // Calculate total remaining balance (principal + fund growth)
    $totalRemaining = $remainingDeposit + $saving->fund_growth_amount;
    
    // Check if this is a full withdrawal (either principal or total balance is near zero)
    $isFullWithdrawal = ($remainingDeposit <= 0.01) || ($totalRemaining <= 0.01);

    // Update status if full withdrawal
    if ($isFullWithdrawal) {
        $saving->saving_status = 'cancelled';
        $saving->maturity_status = 'terminated';
    }

    $saving->save();
    return $saving;
}


}
