<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\EasypaisaUser;
use App\Models\InvestmentLedgerSaving;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use App\Models\CustomerSavingsMaster;

class WithdrawController extends Controller
{


public function withdrawSummary(Request $request)
{
    $request->validate(['msisdn' => 'required|string']);

    // Find customer
    $user = EasypaisaUser::where('user_msisdn', $request->msisdn)->first();

    if (!$user) {
        return response()->json([
            'status' => false,
            'message' => 'Customer not found'
        ], 404);
    }

    // Get the active saving account
    $saving = CustomerSavingsMaster::where('customer_id', $user->id)
        ->where('saving_status', 'on-going')
        ->first();

    if (!$saving) {
        return response()->json([
            'status' => true,
            'message' => 'No active savings found',
            'data' => [
                'msisdn' => $request->msisdn,
                'total_deposit' => 0,
                'total_withdrawal' => 0,
                'available_balance' => 0,
                'total_profit_till_last_month' => 0,
                'is_zakat_applicable' => 0
            ]
        ]);
    }

    // Get the latest ledger entry which contains current balances
    $lastLedgerEntry = InvestmentLedgerSaving::where('saving_id', $saving->id)
        ->orderBy('date_time', 'desc')
        ->first();

    // Calculate total deposits (sum of all deposit transactions)
    $totalDeposit = InvestmentLedgerSaving::where('saving_id', $saving->id)
        ->where('transaction_type', 'deposit')
        ->sum('amount');

    // Calculate total withdrawals (sum of all withdrawal transaction amounts)
    $totalWithdrawal = InvestmentLedgerSaving::where('saving_id', $saving->id)
        ->where('transaction_type', 'withdrawal')
        ->sum('amount');

    return response()->json([
        'status' => true,
        'message' => 'Withdraw summary fetched successfully',
        'data' => [
            'msisdn' => $request->msisdn,
            'total_deposit' => $totalDeposit,
            'total_withdrawal' => $totalWithdrawal,
            'available_balance' => $lastLedgerEntry->net_amount ?? 0,
            'total_profit_till_last_month' => $lastLedgerEntry->profit_amount ?? 0,
            'is_zakat_applicable' => $saving->is_zakat_applicable ?? 0
        ]
    ]);
}



public function withdraw(Request $request)
{
    $request->validate([
        'customer_msisdn' => 'required|string',
        'openid' => 'required|string',
        'profit_amount' => 'required|numeric|min:0',
        'withdraw_amount' => 'required|numeric|min:0',
        'zakat_amount' => 'required|numeric|min:0',
        'declaration_form_submitted' => 'required|boolean',
        'is_refund' => 'sometimes|boolean'
    ]);

    // Fetch customer
    $customer = EasypaisaUser::where([
        ['user_msisdn', '=', $request->customer_msisdn],
        ['open_id', '=', $request->openid],
    ])->first();

    if (!$customer) {
        return response()->json([
            'status' => false,
            'message' => 'Customer not found.',
        ], 404);
    }

    // Fetch saving record
    $saving = CustomerSavingsMaster::where('customer_id', $customer->id)
        ->where('saving_status', 'on-going')
        ->first();

    if (!$saving) {
        return response()->json([
            'status' => false,
            'message' => 'Active saving record not found.',
        ]);
    }

    // Get current balances
    $balance = InvestmentLedgerSaving::where('saving_id', $saving->id)
        ->orderBy('date_time', 'desc')
        ->first();

    // Handle refund case
    if ($request->is_refund) {
        return $this->processRefund($request, $customer, $saving, $balance);
    }

    // Validate withdrawal amounts
    $validation = $this->validateWithdrawalAmounts($request, $balance);
    if ($validation !== true) {
        return $validation;
    }

    DB::beginTransaction();
    try {
        $transactionId = uniqid('txn_');
        $now = now();

        // Calculate actual amounts to process
        $principalAmount = min($request->withdraw_amount, $balance->principal_amount);
        $profitAmount = min($request->profit_amount, $balance->profit_amount);
        
        // Calculate zakat on total withdrawal amount (principal + profit)
        $totalWithdrawalAmount = $principalAmount + $profitAmount;
        $zakatAmount = min($request->zakat_amount, $totalWithdrawalAmount * 0.025); // 2.5% of total withdrawal

        // Create withdrawal record
        $withdrawal = $this->createWithdrawalRecord(
            $customer,
            $saving,
            $transactionId,
            $principalAmount,
            $profitAmount,
            $zakatAmount,
            $balance,
            $now
        );

        // Update saving status and fund growth
        $saving = $this->updateSavingStatus(
            $saving,
            $balance->principal_amount - $principalAmount, // remaining deposit
            $principalAmount, // withdrawn principal amount
            $profitAmount,    // withdrawn profit amount
            $zakatAmount
        );

        DB::commit();

        return response()->json([
            'status' => true,
            'message' => 'Withdrawal processed successfully.',
            'data' => $this->prepareResponseData(
                $transactionId,
                $principalAmount,
                $profitAmount,
                $zakatAmount,
                $balance->principal_amount - $principalAmount,
                $balance->profit_amount - $profitAmount
            )
        ]);

    } catch (\Throwable $e) {
        DB::rollBack();
        return $this->handleError($e);
    }
}

protected function validateWithdrawalAmounts($request, $balance)
{
    // Check if trying to withdraw more principal than available
    if ($request->withdraw_amount > $balance->principal_amount) {
        return response()->json([
            'status' => false,
            'message' => 'Insufficient deposit balance.',
            'available_deposit' => number_format($balance->principal_amount, 2),
        ]);
    }

    // Check if trying to withdraw more profit than available
    if ($request->profit_amount > $balance->profit_amount) {
        return response()->json([
            'status' => false,
            'message' => 'Insufficient profit balance.',
            'available_profit' => number_format($balance->profit_amount, 2),
        ]);
    }

    // Check if zakat is more than 2.5% of total withdrawal amount
    $totalWithdrawal = $request->withdraw_amount + $request->profit_amount;
    $maxZakat = $totalWithdrawal * 0.025;
   // return $maxZakat;
    if ($request->zakat_amount > $maxZakat) {
        return response()->json([
            'status' => false,
            'message' => 'Zakat amount cannot exceed 2.5% of total withdrawal amount.',
            'maximum_zakat_allowed' => number_format($maxZakat, 2),
        ]);
    }

    return true;
}

protected function createWithdrawalRecord($customer, $saving, $transactionId, $principalAmount, $profitAmount, $zakatAmount, $balance, $now)
{
    $grossAmount = $principalAmount + $profitAmount;
    $netAmount = $grossAmount - $zakatAmount;

    return InvestmentLedgerSaving::create([
        'customer_id' => $customer->id,
        'saving_id' => $saving->id,
        'customer_msisdn' => $customer->user_msisdn,
        'transaction_id' => $transactionId,
        'transaction_type' => 'withdrawal',
        'amount' => $netAmount,
        'principal_amount' => $balance->principal_amount - $principalAmount,
        'profit_amount' => $balance->profit_amount - $profitAmount,
        'zakat_amount' => $zakatAmount,
        'date_time' => $now,
        'net_amount' => $balance->net_amount - $netAmount,
        'gross_amount' => $grossAmount,
        'breakdown' => [
            'deposit_amount' => $principalAmount,
            'profit_amount' => $profitAmount,
            'zakat_amount' => $zakatAmount,
            'zakat_rate' => '2.5%',
            'zakat_base' => $principalAmount + $profitAmount,
            'transaction_type' => 'withdrawal'
        ]
    ]);
}

// Keep other methods (updateSavingStatus, prepareResponseData, processRefund, handleError) unchanged

protected function updateSavingStatus($saving, $remainingDeposit, $withdrawnPrincipal, $withdrawnProfit, $zakatAmount)
{
    // Update fund growth (deduct only the profit portion)
    $saving->fund_growth_amount = max(0, $saving->fund_growth_amount - $withdrawnProfit - $withdrawnPrincipal);
    
    // Calculate total remaining balance (principal + fund growth)
    $totalRemaining = $remainingDeposit + $saving->fund_growth_amount;
    
    // Check if this is a full withdrawal (either principal or total balance is near zero)
    $isFullWithdrawal = ($remainingDeposit <= 0.01) || ($totalRemaining <= 0.01);

    // Update status if full withdrawal
    if ($isFullWithdrawal) {
        $saving->saving_status = 'cancelled';
        $saving->maturity_status = 'terminated';
    }

    $saving->save();
    return $saving;
}

protected function prepareResponseData($transactionId, $principalAmount, $profitAmount, $zakatAmount, $remainingDeposit, $remainingProfit)
{
    return [
        'transaction_id' => $transactionId,
        'withdraw_amount' => number_format($principalAmount, 2),
        'profit_used' => number_format($profitAmount, 2),
        'zakat_deducted' => number_format($zakatAmount, 2),
        'available_deposit_after' => number_format($remainingDeposit, 2),
        'available_profit_after' => number_format($remainingProfit, 2),
        'redirect_to' => $remainingDeposit <= 0.01 ? 'start_saving' : 'active_saving',
    ];
}

// Keep your existing processRefund method unchanged

protected function handleError($e)
{
    \Log::error('Transaction Error: '.$e->getMessage());
    return response()->json([
        'status' => false,
        'message' => 'Error processing transaction.',
        'error' => $e->getMessage(),
    ], 500);
}
}








