<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\EasypaisaUser;
use App\Models\PdfReport;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use App\Models\Refund;
use App\Models\Customer;
use Mpdf\Mpdf;
use Smalot\PdfParser\Parser;
use PhpOffice\PhpSpreadsheet\IOFactory;
use App\Models\CustomerSavingsMaster;
use App\Models\KiborRate;
use App\Models\Slab;
use App\Models\DailyReturn;
use Carbon\Carbon;



class PDFController extends Controller
{


    public function generate(Request $request)
{
    // Step 1: Validate the request
    $validated = $request->validate([
        'msisdn' => 'required|string',
        'amount' => 'required|numeric',
        'contribution_term' => 'required|numeric',
    ]);

    $validated['contribution_term'] = (int) ceil($validated['contribution_term'] / 365);

    // Step 2: Fetch user
    $user = EasypaisaUser::where('user_msisdn', $validated['msisdn'])->first();

    if (!$user) {
        return response()->json([
            'status' => false,
            'message' => 'User not found for the given MSISDN.'
        ], 404);
    }

    // Step 3: Calculate values
    $data = $this->calculateIllustrationValues($validated['amount'], $validated['contribution_term'], $user);

    // Step 4: Generate PDF
    try {
        $mpdf = new Mpdf([
            'mode' => 'utf-8',
            'format' => 'A4-L',
            'default_font' => 'notonastaliq',
            'default_font_size' => 10,
            'margin_left' => 10,
            'margin_right' => 10,
            'tempDir' => storage_path('temp')
        ]);

        $mpdf->autoScriptToLang = true;
        $mpdf->autoLangToFont = true;

        $html = view('pdfs.benefit', $data)->render();
        $mpdf->WriteHTML($html);

        // Save PDF
        $filename = 'benefit_' . Str::random(10) . '.pdf';
        $storagePath = "pdfs/{$filename}";

        if (!Storage::disk('public')->exists('pdfs')) {
            Storage::disk('public')->makeDirectory('pdfs');
        }

        Storage::disk('public')->put($storagePath, $mpdf->Output('', 'S'));

        if (!Storage::disk('public')->exists($storagePath)) {
            throw new \Exception("Failed to save PDF file");
        }

        //return $storagePath;

        // Store in database
        $this->storePdfReport($user, $validated['amount'], $validated['contribution_term'], $data, $storagePath);

        // Return response
        return response()->json([
            'status' => 'success',
            'download_url' => asset("storage/app/public/{$storagePath}")
        ]);
    } catch (\Exception $e) {
        return response()->json([
            'status' => 'error',
            'message' => 'PDF generation failed: ' . $e->getMessage()
        ], 500);
    }
}

protected function storePdfReport($user, $amount, $term, $data, $storagePath)
{
    
    
    $lastYearData = end($data['illustrativeValues']);

        $profitAt9 = (float) str_replace(',', '', $lastYearData['9p_surrender']);
        $profitAt13 = (float) str_replace(',', '', $lastYearData['13p_surrender']);

    PdfReport::create([
        'name' => $user->first_name . ' ' . $user->last_name,
        'msisdn' => $user->user_msisdn,
        'amount' => $amount,
        'contribution_term' => $term,
        'sum_covered' => $data['sum_covered_raw'],
        'profit_at_9' => $profitAt9,
        'profit_at_13' => $profitAt13,
        'pdf_path' => "{$storagePath}",
    ]);
}


protected function calculateIllustrationValues($amount, $term, $user)
{
    // Define rate table as array (sorted ascending)
    $rateTable = [
        9999     => ['9' => ['customer' => 3, 'efu' => 1, 'ep' => 5], '13' => ['customer' => 5, 'efu' => 1, 'ep' => 7]],
        49999    => ['9' => ['customer' => 4, 'efu' => 1, 'ep' => 4], '13' => ['customer' => 6, 'efu' => 1, 'ep' => 6]],
        149999   => ['9' => ['customer' => 5, 'efu' => 1, 'ep' => 3], '13' => ['customer' => 7, 'efu' => 1, 'ep' => 5]],
        1000000  => ['9' => ['customer' => 6, 'efu' => 1, 'ep' => 2], '13' => ['customer' => 8, 'efu' => 1, 'ep' => 4]],
    ];

    ksort($rateTable); // ensure slabs are in order

    // Initialize cumulative amounts as strings for BCMath
    $cumulativeAmount9 = (string) $amount;
    $cumulativeAmount13 = (string) $amount;

    $illustrativeValues = [];

    for ($year = 1; $year <= $term; $year++) {
        for ($day = 0; $day < 365; $day++) {
            // Get current slab for 9% product
            foreach ($rateTable as $limit => $rates) {
                if (bccomp($cumulativeAmount9, (string)$limit) <= 0) {
                    $rate9 = $rates['9'];
                    break;
                }
            }

            // Get current slab for 13% product
            foreach ($rateTable as $limit => $rates) {
                if (bccomp($cumulativeAmount13, (string)$limit) <= 0) {
                    $rate13 = $rates['13'];
                    break;
                }
            }

            // Ensure a slab was found
            if (!isset($rate9) || !isset($rate13)) {
                throw new Exception("Amount $amount exceeds rate table limits");
            }

            // Calculate daily rates as floats
            $dailyRate9 = pow(1 + ($rate9['customer'] / 100), 1 / 365) - 1;
            $dailyRate13 = pow(1 + ($rate13['customer'] / 100), 1 / 365) - 1;

            // Convert daily rates to strings with fixed decimal precision (20 decimals)
            $dailyRate9Str = number_format($dailyRate9, 20, '.', '');
            $dailyRate13Str = number_format($dailyRate13, 20, '.', '');

            // Calculate daily compounded amounts using BCMath
            $increment9 = bcmul($cumulativeAmount9, $dailyRate9Str, 14);
            $cumulativeAmount9 = bcadd($cumulativeAmount9, $increment9, 8);

            $increment13 = bcmul($cumulativeAmount13, $dailyRate13Str, 14);
            $cumulativeAmount13 = bcadd($cumulativeAmount13, $increment13, 8);

            // Log daily profit and summed amounts for the first 3 days of each year
            if ($day < 3) {
                error_log("Year $year, Day " . ($day + 1) . ": " .
                    "9% Rate = $dailyRate9Str, 9% Profit = $increment9, 9% Sum = $cumulativeAmount9 | " .
                    "13% Rate = $dailyRate13Str, 13% Profit = $increment13, 13% Sum = $cumulativeAmount13");
            }
        }

        // Store yearly snapshot (formatted for display)
        $illustrativeValues[] = [
            'year' => $year,
            'contributions' => number_format($amount, 0),
            'allocated' => $year === 1 ? number_format($amount, 0) : '-',
            '9p_surrender' => number_format($cumulativeAmount9, 0),
            '9p_death' => number_format($amount * 1.25, 0),
            '13p_surrender' => number_format($cumulativeAmount13, 0),
            '13p_death' => number_format($amount * 1.25, 0),
            '9p_surrender_raw' => round((float)$cumulativeAmount9, 8),
            '13p_surrender_raw' => round((float)$cumulativeAmount13, 8),
        ];
    }

    // Final slab (based on original amount only) to calculate EP profits
    foreach ($rateTable as $limit => $rates) {
        if ($amount <= $limit) {
            $finalRate9 = $rates['9'];
            $finalRate13 = $rates['13'];
            break;
        }
    }

    $sumCovered = $amount * 1.25;
    $profit9 = $amount * ($finalRate9['ep'] / 100);
    $profit13 = $amount * ($finalRate13['ep'] / 100);

    return [
        'name' => $user->first_name . ' ' . $user->last_name,
        'msisdn' => $user->user_msisdn,
        'amount' => number_format($amount, 0),
        'contribution_term' => $term . ' Years',
        'sum_covered' => number_format($sumCovered, 0),
        'profit_at_9' => number_format($profit9, 0),
        'profit_at_13' => number_format($profit13, 0),
        'sum_covered_raw' => $sumCovered,
        'profit_at_9_raw' => $profit9,
        'profit_at_13_raw' => $profit13,
        'dob' => $user->dob ?? 'N/A',
        'clientName' => $user->first_name . ' ' . $user->last_name,
        'participantName' => $user->first_name . ' ' . $user->last_name,
        'referenceNumber' => '092-BMF',
        'age' => $user->dob ? \Carbon\Carbon::parse($user->dob)->age : 'N/A',
        'commencementDate' => \Carbon\Carbon::now()->format('d/m/Y'),
        'validTill' => \Carbon\Carbon::now()->addYears($term)->format('d/m/Y'),
        'paymentMode' => 'Single',
        'paymentModeUrdu' => 'سنگل',
        'planType' => 'Level',
        'planTypeUrdu' => 'سادہ',
        'sumCovered' => number_format($sumCovered, 0),
        'coverageTerm' => $term . ' Years',
        'singleContribution' => number_format($amount, 0),
        'totalSingleContribution' => number_format($amount, 0),
        'accidentalDeathBenefit' => number_format($sumCovered, 0),
        'illustrativeValues' => $illustrativeValues,
        'allocationCharges' => [
            ['year' => 'Year 1 and Onwards', 'rate' => 'Nill']
        ],
    ];
}





}

