<?php

namespace App\Services;

use Illuminate\Support\Facades\Log;

class CryptoService
{
    private $privateKey;

    public function __construct()
    {
        // Load and parse private key from config (easypaisa.private_key)
        $this->privateKey = openssl_pkey_get_private(config('easypaisa.private_key'));

        if (!$this->privateKey) {
            Log::error("CryptoService: Failed to load private key from config.");
        }
    }

    /**
     * Decrypt symmetricKey using RSA private key.
     */
    public function decryptSymmetricKey(string $encryptedKeyBase64): ?string
    {
        $encryptedKey = base64_decode($encryptedKeyBase64);

        if (!$this->privateKey) {
            return null;
        }

        if (!openssl_private_decrypt($encryptedKey, $decrypted, $this->privateKey)) {
            Log::error("CryptoService: RSA decryption failed.");
            return null;
        }

        return $decrypted;
    }

    /**
     * Generate a 16-byte AES key from the randomKey.
     */
    private function generateSecretKey(string $randomKey): string
    {
        return substr(hash('md5', $randomKey, true), 0, 16); // AES-128 key (16 bytes)
    }

    /**
     * Decrypt AES-128-ECB encrypted string.
     */
    public function aesDecrypt(string $base64Content, string $randomKey): ?string
    {
        $key = $this->generateSecretKey($randomKey);
        $ciphertext = base64_decode($base64Content);

        $decrypted = openssl_decrypt(
            $ciphertext,
            'aes-128-ecb',
            $key,
            OPENSSL_RAW_DATA | OPENSSL_ZERO_PADDING
        );

        \Log::info('Testing:', ['data' => $decrypted]);
        return $decrypted ? trim($decrypted) : null;
    }
}
