<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\EasypaisaUser;
use App\Models\InvestmentLedgerSaving;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use App\Models\CustomerSavingsMaster;
use App\Models\RefundCases;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use App\Models\InsuranceData;

class WithdrawController extends Controller
{

//Working Code Breakage Period  
// public function withdrawSummary(Request $request)
// {
//     $request->validate(['msisdn' => 'required|string']);

//     // Find customer
//     $user = EasypaisaUser::where('user_msisdn', $request->msisdn)->first();

//     if (!$user) {
//         return response()->json([
//             'status' => false,
//             'message' => 'Customer not found'
//         ], 404);
//     }

//     // Get the active saving account
//     $saving = CustomerSavingsMaster::where('customer_id', $user->id)
//         ->where('saving_status', 'on-going')
//         ->first();

//     if (!$saving) {
//         return response()->json([
//             'status' => true,
//             'message' => 'No active savings found',
//             'data' => [
//                 'msisdn' => $request->msisdn,
//                 'total_deposit' => 0,
//                 'total_withdrawal' => 0,
//                 'available_balance' => 0,
//                 'total_profit_till_last_month' => 0,
//                 'is_zakat_applicable' => 0,
//                 'zakat_amount' => 0
//             ]
//         ]);
//     }

//     // Get the latest ledger entry which contains current balances
//     $lastLedgerEntry = InvestmentLedgerSaving::where('saving_id', $saving->id)
//         ->orderBy('date_time', 'desc')
//         ->first();

//     $netAmount = $lastLedgerEntry->net_amount ?? 0;
//     $zakatAmount = $lastLedgerEntry->zakat_amount ?? 0;

//     // Calculate available balance excluding zakat
//     $availableBalance = $netAmount - $zakatAmount;

//     // Calculate total deposits (sum of all deposit transactions)
//     $totalDeposit = InvestmentLedgerSaving::where('saving_id', $saving->id)
//         ->where('transaction_type', 'deposit')
//         ->sum('amount');

//     // Calculate total withdrawals (sum of all withdrawal transaction amounts)
//     $totalWithdrawal = InvestmentLedgerSaving::where('saving_id', $saving->id)
//         ->where('transaction_type', 'withdrawal')
//         ->sum('amount');

//     return response()->json([
//         'status' => true,
//         'message' => 'Withdraw summary fetched successfully',
//         'data' => [
//             'msisdn' => $request->msisdn,
//             'total_deposit' => $totalDeposit,
//             'total_withdrawal' => $totalWithdrawal,
//             'available_balance' => round($availableBalance, 2),
//             'total_profit_till_last_month' => $lastLedgerEntry->profit_amount ?? 0,
//             'is_zakat_applicable' => $saving->is_zakat_applicable ?? 0,
//             'zakat_amount' => round($zakatAmount, 2)
//         ]
//     ]);
// }


public function withdrawSummary(Request $request)
{
    $request->validate(['msisdn' => 'required|string']);

    // Find customer
    $user = EasypaisaUser::where('user_msisdn', $request->msisdn)->first();

    if (!$user) {
        return response()->json([
            'status' => false,
            'message' => 'Customer not found'
        ], 404);
    }

    // Get the active saving account
    $saving = CustomerSavingsMaster::where('customer_id', $user->id)
        ->where('saving_status', 'on-going')
        ->first();

    if (!$saving) {
        return response()->json([
            'status' => true,
            'message' => 'No active savings found',
            'data' => [
                'msisdn' => $request->msisdn,
                'total_deposit' => 0,
                'total_withdrawal' => 0,
                'available_balance' => 0,
                'total_profit_till_last_month' => 0,
                'is_zakat_applicable' => 0,
                'zakat_amount' => 0
            ]
        ]);
    }

    // Get the latest ledger entry for current balances
    $lastLedgerEntry = InvestmentLedgerSaving::where('saving_id', $saving->id)
        ->orderBy('date_time', 'desc')
        ->first();

    $netAmount = $lastLedgerEntry->net_amount ?? 0;
    $zakatAmount = $lastLedgerEntry->zakat_amount ?? 0;

    // Calculate available balance excluding zakat
    $availableBalance = $netAmount - $zakatAmount;

    // Calculate total deposits
    $totalDeposit = InvestmentLedgerSaving::where('saving_id', $saving->id)
        ->where('transaction_type', 'deposit')
        ->sum('amount');

    // Calculate total withdrawals
    $totalWithdrawal = InvestmentLedgerSaving::where('saving_id', $saving->id)
        ->where('transaction_type', 'withdrawal')
        ->sum('amount');

    // Get the first day of the current month
    $startOfCurrentMonth = now()->startOfMonth();

    // Sum of profit till last month (exclude current month entries)
    $totalProfitTillLastMonth = InvestmentLedgerSaving::where('saving_id', $saving->id)
        ->where('date_time', '<', $startOfCurrentMonth)
        ->sum('profit_amount');

    return response()->json([
        'status' => true,
        'message' => 'Withdraw summary fetched successfully',
        'data' => [
            'msisdn' => $request->msisdn,
            'total_deposit' => $totalDeposit,
            'total_withdrawal' => $totalWithdrawal,
            'available_balance' => round($availableBalance, 2),
            'total_profit_till_last_month' => round($totalProfitTillLastMonth, 2),
            'is_zakat_applicable' => $saving->is_zakat_applicable ?? 0,
            'zakat_amount' => round($zakatAmount, 2)
        ]
    ]);
}


//Working Code Non-Breakage Period 
// public function withdraw(Request $request)
// {
//     $request->validate([
//         'customer_msisdn' => 'required|string',
//         'openid' => 'required|string',
//         'profit_amount' => 'required|numeric|min:0',
//         'withdraw_amount' => 'required|numeric|min:0',
//         'zakat_amount' => 'required|numeric|min:0',
//         'declaration_form_submitted' => 'required|boolean',
//         'is_refund' => 'sometimes|boolean'
//     ]);

//     $customer = EasypaisaUser::where([
//         ['user_msisdn', '=', $request->customer_msisdn],
//         ['open_id', '=', $request->openid],
//     ])->first();

//     if (!$customer) {
//         return response()->json(['status' => false, 'message' => 'Customer not found.'], 404);
//     }

//     $saving = CustomerSavingsMaster::where('customer_id', $customer->id)
//         ->where('saving_status', 'on-going')
//         ->first();

//     if (!$saving) {
//         return response()->json(['status' => false, 'message' => 'Active saving record not found.']);
//     }

//     $balance = InvestmentLedgerSaving::where('saving_id', $saving->id)
//         ->orderBy('date_time', 'desc')
//         ->first();

//     if ($request->is_refund) {
//         return $this->processRefund($request, $customer, $saving, $balance);
//     }

//     // Handle declaration-based exemption
//     $zakatAmount = 0;
//     $zakatOption = '3';

//     if ($request->declaration_form_submitted) {
//         $zakatOption = '2';
//     } elseif ($request->zakat_amount > 0) {
//         $zakatOption = '1';
//         $zakatAmount = $request->zakat_amount;
//     }
    
    

//     if ($request->declaration_form_submitted) {
//         RefundCases::create([
//             'investment_master_id' => $saving->id,
//             'refund_amount' => $request->withdraw_amount,
//             'refund_request_date' => now(),
//             'processing_fee_deducted' => 0,
//             'refund_processed_date' => null,
//             'status' => 'pending',
//             'type' => 'Under-Processing',
//             'zakat_amount' => 0,
//             'profit_amount' => $request->profit_amount,
//             'saving_id' => $saving->id,
//             'customer_id' => $customer->id,
//         ]);

//         $saving->update(['declaration_form_submitted' => true]);

//         return response()->json([
//             'status' => true,
//             'message' => 'Refund request submitted successfully.',
//             'redirect_to' => 'active_saving_form'
//         ]);
//     }

//     $validation = $this->validateWithdrawalAmounts($request, $balance);
//     if ($validation !== true) {
//         return $validation;
//     }

//     DB::beginTransaction();
//     try {
//         $transactionId = uniqid('txn_');
//         $now = now();

//         $principalAmount = min($request->withdraw_amount, $balance->principal_amount);
//         $profitAmount = min($request->profit_amount, $balance->profit_amount);

//         $totalWithdrawalAmount = $principalAmount + $profitAmount;

//         // EFU customer_id
//         $insuranceData = InsuranceData::where('saving_id', $saving->id)->first();
//         if (!$insuranceData || !$insuranceData->customer_id) {
//             DB::rollBack();
//             return response()->json(['status' => false, 'message' => 'EFU customer ID not found.']);
//         }

//         $surrenderType = ($balance->principal_amount <= $principalAmount && $balance->profit_amount <= $profitAmount) ? 'full' : 'partial';

//         // EFUL Payload
//         $efuPayload = [
//             'customer_id' => $insuranceData->eful_policy_number,
//             'surrender_type' => $surrenderType,
//             'principle_amount' => round($principalAmount, 2),
//             'profit_amount' => round($profitAmount, 2),
//             'zakat_amount' => round($zakatAmount, 2),
//             'zakat_option' => $zakatOption,
//             'hypo_rate' => "11",
//             'hypo_intrest' => "0.0"
//         ];

//                 $efuResponse = Http::withHeaders([
//                 'Authorization' => 'Bearer XXXXAAA123EPDIGITAKAFUL',
//                 'channelcode' => 'EPDIGITAKAFUL',
//                 'Content-Type' => 'application/json',
                
//             ])
//             ->post('https://api.efulife.com/epay/digi/tkf/surrender', $efuPayload);

//         if (!$efuResponse->successful()) {
//             Log::error('EFU API Error', [
//                 'payload' => $efuPayload,
//                 'response_status' => $efuResponse->status(),
//                 'response_body' => $efuResponse->body(),
//             ]);
//             DB::rollBack();
//             return response()->json([
//                 'status' => false,
//                 'message' => 'EFU API request failed. Please try again later.',
//             ]);
//         }
        
//         Log::info('EFU API Success', [
//     'payload' => $efuPayload,
//     'response_status' => $efuResponse->status(),
//     'response_body' => $efuResponse->body(),
// ]);

//         // Store & update
//         $this->createWithdrawalRecord(
//             $customer,
//             $saving,
//             $transactionId,
//             $principalAmount,
//             $profitAmount,
//             $zakatAmount,
//             $balance,
//             $now
//         );

//         $this->updateSavingStatus(
//             $saving,
//             $balance->principal_amount - $principalAmount,
//             $principalAmount,
//             $profitAmount,
//             $zakatAmount
//         );

//         DB::commit();

//         return response()->json([
//             'status' => true,
//             'message' => 'Withdrawal processed successfully.',
//             'data' => $this->prepareResponseData(
//                 $transactionId,
//                 $principalAmount,
//                 $profitAmount,
//                 $zakatAmount,
//                 $balance->principal_amount - $principalAmount,
//                 $balance->profit_amount - $profitAmount
//             )
//         ]);

//     } catch (\Throwable $e) {
//         DB::rollBack();
//         return $this->handleError($e);
//     }
// }

public function withdraw(Request $request)
{
    $request->validate([
        'customer_msisdn' => 'required|string',
        'openid' => 'required|string',
        'profit_amount' => 'required|numeric|min:0',
        'withdraw_amount' => 'required|numeric|min:0',
        'zakat_amount' => 'required|numeric|min:0',
        'declaration_form_submitted' => 'required|boolean',
        'is_refund' => 'sometimes|boolean'
    ]);

    $customer = EasypaisaUser::where([
        ['user_msisdn', '=', $request->customer_msisdn],
        ['open_id', '=', $request->openid],
    ])->first();

    if (!$customer) {
        return response()->json(['status' => false, 'message' => 'Customer not found.'], 404);
    }

    $saving = CustomerSavingsMaster::where('customer_id', $customer->id)
        ->where('saving_status', 'on-going')
        ->first();

    if (!$saving) {
        return response()->json(['status' => false, 'message' => 'Active saving record not found.']);
    }

    $balance = InvestmentLedgerSaving::where('saving_id', $saving->id)
        ->orderBy('date_time', 'desc')
        ->first();

    if ($request->is_refund) {
        return $this->processRefund($request, $customer, $saving, $balance);
    }

    // Handle declaration-based exemption
    $zakatAmount = 0;
    $zakatOption = '3';

    if ($request->declaration_form_submitted) {
        $zakatOption = '2';
    } elseif ($request->zakat_amount > 0) {
        $zakatOption = '1';
        $zakatAmount = $request->zakat_amount;
    }

    // === Withdrawal Conditions ===
    $isSameMonthAsStart = $saving->created_at->format('Y-m') === now()->format('Y-m');

    if ($isSameMonthAsStart) {
        if ($request->profit_amount > 0) {
            return response()->json([
                'status' => false,
                'message' => 'Profit cannot be withdrawn in the same month as saving start. Only principal is allowed.'
            ]);
        }

        $surrenderType = 'full';
    } else {
        // Only allow profit from before the current month
        $currentMonthStart = now()->startOfMonth();

        $lastMonthProfit = InvestmentLedgerSaving::where('saving_id', $saving->id)
            ->where('transaction_type', 'deposit') // or however profit is stored
            ->where('date_time', '<', $currentMonthStart)
            ->sum('profit_amount');

        if ($request->profit_amount > $lastMonthProfit) {
            return response()->json([
                'status' => false,
                'message' => 'You can only withdraw profit up to last month.',
                'maximum_profit_allowed' => number_format($lastMonthProfit, 2),
            ]);
        }

        $surrenderType = ($balance->principal_amount <= $request->withdraw_amount &&
                          $balance->profit_amount <= $request->profit_amount) ? 'full' : 'partial';
    }

    // Declaration refund handling
    if ($request->declaration_form_submitted) {
        RefundCases::create([
            'investment_master_id' => $saving->id,
            'refund_amount' => $request->withdraw_amount,
            'refund_request_date' => now(),
            'processing_fee_deducted' => 0,
            'refund_processed_date' => null,
            'status' => 'pending',
            'type' => 'Under-Processing',
            'zakat_amount' => 0,
            'profit_amount' => $request->profit_amount,
            'saving_id' => $saving->id,
            'customer_id' => $customer->id,
        ]);

        $saving->update(['declaration_form_submitted' => true]);

        return response()->json([
            'status' => true,
            'message' => 'Refund request submitted successfully.',
            'redirect_to' => 'active_saving_form'
        ]);
    }

    // Validate withdrawal limits
    $validation = $this->validateWithdrawalAmounts($request, $balance);
    if ($validation !== true) {
        return $validation;
    }

    DB::beginTransaction();
    try {
        $transactionId = uniqid('txn_');
        $now = now();

        $principalAmount = min($request->withdraw_amount, $balance->principal_amount);
        $profitAmount = min($request->profit_amount, $balance->profit_amount);

        $totalWithdrawalAmount = $principalAmount + $profitAmount;

        // EFU customer_id check
        $insuranceData = InsuranceData::where('saving_id', $saving->id)->first();
        if (!$insuranceData || !$insuranceData->customer_id) {
            DB::rollBack();
            return response()->json(['status' => false, 'message' => 'EFU customer ID not found.']);
        }

        // EFU payload
        $efuPayload = [
            'customer_id' => $insuranceData->eful_policy_number,
            'surrender_type' => $surrenderType,
            'principle_amount' => round($principalAmount, 2),
            'profit_amount' => round($profitAmount, 2),
            'zakat_amount' => round($zakatAmount, 2),
            'zakat_option' => $zakatOption,
            'hypo_rate' => "11",
            'hypo_intrest' => "0.0"
        ];
        
        //dd($efuPayload);

        $efuResponse = Http::withHeaders([
            'Authorization' => 'Bearer XXXXAAA123EPDIGITAKAFUL',
            'channelcode' => 'EPDIGITAKAFUL',
            'Content-Type' => 'application/json',
        ])->post('https://api.efulife.com/epay/digi/tkf/surrender', $efuPayload);

        if (!$efuResponse->successful()) {
            Log::error('EFU API Error', [
                'payload' => $efuPayload,
                'response_status' => $efuResponse->status(),
                'response_body' => $efuResponse->body(),
            ]);
            DB::rollBack();
            return response()->json([
                'status' => false,
                'message' => 'EFU API request failed. Please try again later.',
            ]);
        }

        Log::info('EFU API Success', [
            'payload' => $efuPayload,
            'response_status' => $efuResponse->status(),
            'response_body' => $efuResponse->body(),
        ]);

        $this->createWithdrawalRecord(
            $customer,
            $saving,
            $transactionId,
            $principalAmount,
            $profitAmount,
            $zakatAmount,
            $balance,
            $now
        );

        $this->updateSavingStatus(
            $saving,
            $balance->principal_amount - $principalAmount,
            $principalAmount,
            $profitAmount,
            $zakatAmount
        );

        DB::commit();

        return response()->json([
            'status' => true,
            'message' => 'Withdrawal processed successfully.',
            'data' => $this->prepareResponseData(
                $transactionId,
                $principalAmount,
                $profitAmount,
                $zakatAmount,
                $balance->principal_amount - $principalAmount,
                $balance->profit_amount - $profitAmount
            )
        ]);
    } catch (\Throwable $e) {
        DB::rollBack();
        return $this->handleError($e);
    }
}


protected function validateWithdrawalAmounts($request, $balance)
{
    // Check if trying to withdraw more principal than available
    if ($request->withdraw_amount > $balance->principal_amount) {
        return response()->json([
            'status' => false,
            'message' => 'Insufficient deposit balance.',
            'available_deposit' => number_format($balance->principal_amount, 2),
        ]);
    }

    // Check if trying to withdraw more profit than available
    if ($request->profit_amount > $balance->profit_amount) {
        return response()->json([
            'status' => false,
            'message' => 'Insufficient profit balance.',
            'available_profit' => number_format($balance->profit_amount, 2),
        ]);
    }

    // Check if zakat is more than 2.5% of total withdrawal amount
    $totalWithdrawal = $request->withdraw_amount + $request->profit_amount;
    $maxZakat = $totalWithdrawal * 0.025;
   // return $maxZakat;
    if ($request->zakat_amount > $maxZakat) {
        return response()->json([
            'status' => false,
            'message' => 'Zakat amount cannot exceed 2.5% of total withdrawal amount.',
            'maximum_zakat_allowed' => number_format($maxZakat, 2),
        ]);
    }

    return true;
}

protected function createWithdrawalRecord($customer, $saving, $transactionId, $principalAmount, $profitAmount, $zakatAmount, $balance, $now)
{
    $grossAmount = $principalAmount + $profitAmount;
    $netAmount = $grossAmount - $zakatAmount;

    return InvestmentLedgerSaving::create([
        'customer_id' => $customer->id,
        'saving_id' => $saving->id,
        'customer_msisdn' => $customer->user_msisdn,
        'transaction_id' => $transactionId,
        'transaction_type' => 'withdrawal',
        'amount' => $netAmount,
        'principal_amount' => $balance->principal_amount - $principalAmount,
        'profit_amount' => $balance->profit_amount - $profitAmount,
        'zakat_amount' => $zakatAmount,
        'date_time' => $now,
        'net_amount' => $balance->net_amount - $netAmount,
        'gross_amount' => $grossAmount,
        'breakdown' => [
            'deposit_amount' => $principalAmount,
            'profit_amount' => $profitAmount,
            'zakat_amount' => $zakatAmount,
            'zakat_rate' => '2.5%',
            'zakat_base' => $principalAmount + $profitAmount,
            'transaction_type' => 'withdrawal'
        ]
    ]);
}

// Keep other methods (updateSavingStatus, prepareResponseData, processRefund, handleError) unchanged

protected function updateSavingStatus($saving, $remainingDeposit, $withdrawnPrincipal, $withdrawnProfit, $zakatAmount)
{
    // Update fund growth (deduct only the profit portion)
    $saving->fund_growth_amount = max(0, $saving->fund_growth_amount - $withdrawnProfit - $withdrawnPrincipal);
    
    // Calculate total remaining balance (principal + fund growth)
    $totalRemaining = $remainingDeposit + $saving->fund_growth_amount;
    
    // Check if this is a full withdrawal (either principal or total balance is near zero)
    $isFullWithdrawal = ($remainingDeposit <= 0.01) || ($totalRemaining <= 0.01);

    // Update status if full withdrawal
    if ($isFullWithdrawal) {
        $saving->saving_status = 'cancelled';
        $saving->maturity_status = 'terminated';
    }

    $saving->save();
    return $saving;
}

protected function prepareResponseData($transactionId, $principalAmount, $profitAmount, $zakatAmount, $remainingDeposit, $remainingProfit)
{
    return [
        'transaction_id' => $transactionId,
        'withdraw_amount' => number_format($principalAmount, 2),
        'profit_used' => number_format($profitAmount, 2),
        'zakat_deducted' => number_format($zakatAmount, 2),
        'available_deposit_after' => number_format($remainingDeposit, 2),
        'available_profit_after' => number_format($remainingProfit, 2),
        'redirect_to' => $remainingDeposit <= 0.01 ? 'start_saving' : 'active_saving',
    ];
}

// Keep your existing processRefund method unchanged

protected function handleError($e)
{
    \Log::error('Transaction Error: '.$e->getMessage());
    return response()->json([
        'status' => false,
        'message' => 'Error processing transaction.',
        'error' => $e->getMessage(),
    ], 500);
}
}








